﻿// Copyright PS-Tech B.V. All Rights Reserved.

using System;
using System.Runtime.InteropServices;
using System.Threading;
using PSTech.Pstsdk;

namespace PSTech.SharedMemory
{
    /// <summary>Shared memory example of the PST SDK</summary>
    /// <remarks>
    /// This example shows how to activate the shared memory communication pipeline
    /// that enables communication of the PST Client application through the PST SDK.
    /// Note that for the shared memory pipeline to function, the application has to
    /// run with elevated access (administrator rights). After enabling shared memory,
    /// the PST Client application can be used to download calibration information and
    /// manage tracking targets.
    /// </remarks>
    public class SharedMemory
    {
        //Registering control handler function.
        [DllImport("Kernel32")]
        private static extern bool SetConsoleCtrlHandler(ConsoleEventHandler handler, bool add);

        private delegate bool ConsoleEventHandler(CtrlType sig);
        static ConsoleEventHandler ConsoleCtrlEventHandler;

        public enum CtrlType
        {
            CtrlCEvent = 0,
            CtrlBreakEvent = 1,
            CtrlCloseEvent = 2,
            CtrlLogOffEvent = 5,
            CtrlShutdownEvent = 6,
        }

        /// <summary>Define handler functions required to ensure a clean shutdown of the PST Tracker when the application is terminated..</summary>
        /// <param name="sig">Signal type</param>
        /// <returns>Signal handling status</returns>
        private static bool ConsoleHandler(CtrlType sig)
        {
            Exithandler((int)sig);
            return true;
        }

        /// <summary>
        /// Implement the exit handler to shut-down the PST Tracker connection on application termination.
        /// </summary>
        /// <param name="sig">Signal type.</param>
        static void Exithandler(int sig)
        {
            Tracker.DisableSharedMemory();
            Tracker.Shutdown();
            Environment.Exit(0);
        }

        static int Main(string[] args)
        {
            // Register the exit handler with the application
            ConsoleCtrlEventHandler += new ConsoleEventHandler(ConsoleHandler);
            SetConsoleCtrlHandler(ConsoleCtrlEventHandler, true);

            // Implement error handling of PSTech.TrackerException exceptions to prevent
            // improper PST Tracker shutdown on errors.
            try
            {
                // Create an instance of the Tracker object using the default configuration path and file names.
                Tracker tracker = new Tracker();

                // Start the tracker server.
                tracker.StartTracker();

                // Enable the shared memory pipeline to enable client connections.
                Tracker.EnableSharedMemory();

                Console.Write("Shared memory server initialized. Start the PST Client application to create a connection.\n");

                // Wait for one minute before terminating this application.
                Thread.Sleep(60000);

                // Disable the shared memory pipeline. If the PST Client is still connected at this point,
                // it will try to reconnect and otherwise exit.
                Tracker.DisableSharedMemory();
            }
            catch (Exception e)
            {
                Console.Write(e + "\n");
                // Handle the error.
            }
            finally
            {
                // Make sure that the connection to the PST Tracker is shut down properly.
                Tracker.Shutdown();
            }

            return 0;
        }
    }
}
