﻿// Copyright PS-Tech B.V. All Rights Reserved.

using System;
using System.Runtime.InteropServices;
using System.Threading;
using PSTech.Pstsdk;

namespace PSTech.RestServer
{
    /// <summary>Rest server example of the PST SDK</summary>
    /// <remarks>
    /// This example shows how to enable the REST server using the PST SDK.
    /// The REST server enables network-based access to the PST Tracker using
    /// the HTTP protocol. GET and POST requests can be made to the server to
    /// send and receive data and change parameters. The REST interface offers
    /// a programming language independent interface to the PST Tracker.
    /// Besides accessing the REST server directly using a browser, a networking
    /// library like Curl can be used to interface with the server programatically.
    /// </remarks>
    public class RestServer
    {
        //Registering control handler function.
        [DllImport("Kernel32")]
        private static extern bool SetConsoleCtrlHandler(ConsoleEventHandler handler, bool add);

        private delegate bool ConsoleEventHandler(CtrlType sig);
        static ConsoleEventHandler ConsoleCtrlEventHandler;

        public enum CtrlType
        {
            CtrlCEvent = 0,
            CtrlBreakEvent = 1,
            CtrlCloseEvent = 2,
            CtrlLogOffEvent = 5,
            CtrlShutdownEvent = 6,
        }

        /// <summary>Define handler functions required to ensure a clean shutdown of the PST Tracker when the application is terminated.</summary>
        /// <param name="sig">Signal type</param>
        /// <returns>Signal handling status</returns>
        private static bool ConsoleHandler(CtrlType sig)
        {
            Exithandler((int)sig);
            return true;
        }

        /// <summary>
        /// Implement the exit handler to shut-down the PST Tracker connection on application termination.
        /// </summary>
        /// <param name="sig">Signal type.</param>
        static void Exithandler(int sig)
        {
            Tracker.DisableRestServer();
            Tracker.Shutdown();
            Environment.Exit(0);
        }

        static int Main(string[] args)
        {
            // Register the exit handler with the application
            ConsoleCtrlEventHandler += new ConsoleEventHandler(ConsoleHandler);
            SetConsoleCtrlHandler(ConsoleCtrlEventHandler, true);

            // Implement error handling of PSTech.TrackerException exceptions to prevent
            // improper PST Tracker shutdown on errors.
            try
            {
                // Create an instance of the Tracker object using the default configuration path and file names.
                Tracker tracker = new Tracker();

                // Start the tracker server.
                tracker.StartTracker();

                // Enable the REST server. To check if the REST server is started correctly, browse to
                // http://localhost:7278/PSTapi/SystemCheck
                // In order to use 127.0.0.1 an an address on Windows 7, execute the following command
                // on an elevated command prompt to allow communication to this address:
                // "netsh http add urlacl url=http://127.0.0.1:7278/ user=EVERYONE listen=yes delegate=no"
                Tracker.EnableRestServer("localhost", "7278");

                Console.Write("PST REST server enabled. See the PST SDK Manual for example commands.\n");

                // Wait for one minute before terminating this application.
                Thread.Sleep(60000);

                // Disable the REST server
                Tracker.DisableRestServer();
            }
            catch (Exception e)
            {
                Console.Write(e + "\n");
                // Handle the error.
            }
            finally
            {
                // Make sure that the connection to the PST Tracker is shut down properly.
                Tracker.Shutdown();
            }

            return 0;
        }
    }
}
