from ctypes import CDLL, c_char_p
from setuptools import setup, find_packages
from logging import error

import os
import glob
import shutil
import platform

class BuildSupport:
    runtime_dir = os.path.join("..", "Redist")

    @staticmethod
    def make():
        if platform.system() == "Windows":
            return BuildSupportWindows()
        elif platform.system() == "Linux":
            return BuildSupportLinux()
        else:
            error("Unsupported platform")

    def copy_runtime(self):
        # Retrieve paths
        package_dir = os.path.join(".", "pstech", "pstsdk", "lib")

        # Copy all runtime files to the temporary package directory
        for src in self.RuntimeFiles:
            dst = os.path.abspath(os.path.join(package_dir))
            if not os.path.exists(dst):
                os.makedirs(dst)

            src = os.path.join(self.runtime_dir, src)
            for f in sorted(glob.glob(src)):
                print("Copy %s => %s" % (f, dst))
                shutil.copy(f, dst)

    def get_package_data(self):
        # Return the data for the pstsdk package
        return ["lib/*"]

class BuildSupportWindows(BuildSupport):
    RuntimeFiles = [
        "pstsdk_c.dll",
        "pstsdk_cpp.dll",
        "nlopt.dll",
        "Freeimage.dll",
        "PocoFoundation.dll",
        "PocoJSON.dll",
        "PocoNet.dll"
    ]

    def update_linking(self):
        # Not necessary for Windows
        pass

class BuildSupportLinux(BuildSupport):
    c_runtime_name = "libpstsdk_c.so"
    cpp_runtime_name = "libpstsdk_cpp.so"

    RuntimeFiles = [
        c_runtime_name + ".*.*.*",
        cpp_runtime_name + ".*.*.*",
        "libnlopt.so*",
        "libfreeimage-*.so"
    ]

    def update_linking(self):
        # Retrieve paths
        package_dir = os.path.abspath(os.path.join(".", "pstech", "pstsdk", "lib"))
              
        # Retrieve version from the shared lib
        c_dll = os.path.join(self.runtime_dir, self.c_runtime_name)
        c_api = CDLL(glob.glob(c_dll)[0])
        c_api.pst_sdk_get_version.restype = c_char_p
        sdk_version = c_api.pst_sdk_get_version().decode('UTF-8')
        major, minor, patch, revision = sdk_version.split(".")

        # Find the correct files and update them
        cpp_so_file = self.cpp_runtime_name + "." + major + "." + minor + "." + patch
        c_so_file = os.path.join(package_dir, self.c_runtime_name + "." + major + "." + minor + "." + patch)
        new_c_so_file = os.path.join(package_dir, self.c_runtime_name)
        
        # Patch the needed dependency so it points to the correct lib
        # patchelf --replace-needed libpstsdk_cpp.so.0 libpstsdk_cpp.0.0.0 /tmp/pip-req-build-{}/libpstsdk_c.so
        os.system("patchelf --replace-needed " + self.cpp_runtime_name + "." + major + " " + cpp_so_file + " " + c_so_file)
        
        # Rename libpstsdk_c.so.0.0.0 to libpstsdk_c.so
        os.rename(c_so_file, new_c_so_file)

if __name__ == "__main__":
    # Setup runtime files before packaging
    bs = BuildSupport.make()
    bs.copy_runtime()
    bs.update_linking()
    package_data = bs.get_package_data()

    setup()
