/**
 * minimal.cpp
 * Copyright PS-Tech B.V. All Rights Reserved.
 *
 * This is a bare minimum example showing how to connect to the PST SDK and how to 
 * register a listener to receive data.
 * 
 * Note that for simplicity reasons, this 
 * example does not take the recommended safety precautions to shut down the PST
 * Tracker when the application shuts down. In an actual implementation, please
 * follow the safety precautions recommended in the "Safe Tracker Termination"
 * section of the "Using PST SDK" page of the PST SDK manual. Examples of how to implement
 * the safety precautions can be found in the other examples.
 *
 */

#include <thread>
#include <chrono>

#include "pstsdk_cpp.h"
#include "TrackerExceptions.h"

/*
 * Implementation of the PSTech::pstsdk::Listener class to receive tracking data.
 * The OnTrackerData() callback function receives the data as soon as it becomes
 * available and prints the tracking target pose to the command line.
 */
class MyListener : public PSTech::pstsdk::Listener
{
    virtual void OnTrackerData(const PSTech::pstsdk::TrackerData& td)
    {
        // Do something with the tracker data in td
    }
} listener;

int main(int argc, char *argv[])
{
    // Implement error handling of PSTech::TrackerException exceptions to prevent 
    // improper PST Tracker shutdown on errors.
    try
    {
        // Create an instance of the Tracker object using the default configuration path and file names.
#ifdef WIN32
        PSTech::pstsdk::Tracker pst;
#else
        // On Linux, specify the type of grabber that needs to be used as the last parameter: 
        // "basler_ace" for PST HD or "basler_dart" for PST Pico
        PSTech::pstsdk::Tracker pst("","config.cfg","models.db",argv[1]);
#endif
        
        // Register the listener object to the tracker server.
        pst.AddTrackerListener(&listener);

        // Start the tracker server.
        pst.Start();
        
        // Wait for 10 seconds, allowing for the detection of tracking targets.
        std::this_thread::sleep_for(std::chrono::seconds(10));
    }
    catch (PSTech::TrackerException &e)
    {
        // Do something with the error.
    }

    // Make sure that the connection to the PST Tracker is shut down properly.
    PSTech::pstsdk::Tracker::Shutdown();

    return 0;
}