﻿// Copyright PS-Tech B.V. All Rights Reserved.

using System;
using System.Runtime.InteropServices;
using System.Threading;
using PSTech.Pstsdk;

namespace PsTech.Exposure
{
    /// <summary>Exposure example of the PST SDK</summary>
    /// <remarks>
    /// This example shows how to adjust exposure settings using the PST SDK. It shows how to
    /// change exposure settings based on frame rate and the available exposure range for a
    /// certain PST Tracker at a certain frame rate.
    /// </remarks>
    public class Exposure
    {
        //Registering control handler function.
        [DllImport("Kernel32")]
        private static extern bool SetConsoleCtrlHandler(ConsoleEventHandler handler, bool add);

        private delegate bool ConsoleEventHandler(CtrlType sig);
        static ConsoleEventHandler ConsoleCtrlEventHandler;

        public enum CtrlType
        {
            CtrlCEvent = 0,
            CtrlBreakEvent = 1,
            CtrlCloseEvent = 2,
            CtrlLogOffEvent = 5,
            CtrlShutdownEvent = 6,
        }

        /// <summary>Define handler functions required to ensure a clean shutdown of the PST Tracker when the application is terminated.</summary>
        /// <param name="sig">Signal type</param>
        /// <returns>Signal handling status</returns>
        private static bool ConsoleHandler(CtrlType sig)
        {
            Exithandler((int)sig);
            return true;
        }

        // Control variable for main loop.
        static bool Running = true;

        // Number of data points to grab before application termination.
        static int NumberOfSamplesToGrab = 1000;
        static int SamplesGrabbed = 0;

        /// Implementation of the PSTech.Pstsdk.Tracker.TrackerListener class to receive tracking data.
        class MyTrackerListener : TrackerListener
        {
            /// <summary>Implementation of a tracker data callback function</summary>
            /// <remarks>
            /// The OnTrackerData() callback function receives the data as soon as it becomes
            /// available.
            /// </remarks>
            /// <param name="data">Object containing tracking information retrieved from tracker</param>
            /// <param name="status">Status message reported by the tracker.</param>
            public override void OnTrackerData(TrackerData data, ErrorStatus status)
            {
                if (SamplesGrabbed++ >= NumberOfSamplesToGrab)
                {
                    Running = false;
                }

                // Do something with the received data.
            }
        }

        /// <summary>
        /// Implement the exit handler to shut-down the PST Tracker connection on application termination.
        /// </summary>
        /// <param name="sig">Signal type.</param>
        static void Exithandler(int sig)
        {
            Tracker.Shutdown();
            Running = false;
        }
        static int Main(string[] args)
        {
            // Register the exit handler with the application
            ConsoleCtrlEventHandler += new ConsoleEventHandler(ConsoleHandler);
            SetConsoleCtrlHandler(ConsoleCtrlEventHandler, true);

            // Implement error handling of PSTech.TrackerException exceptions to prevent
            // improper PST Tracker shutdown on errors.
            try
            {
                // Create an instance of the Tracker object using the default configuration path and file names.
                Tracker tracker = new Tracker();

                // Print version number of the tracker server being used.
                Console.Write("Running PST Server version " + tracker.GetVersionInfo() + "\n");

                //Create tracker listener with callback functions for data and/or mode updates.
                TrackerListener listener = new MyTrackerListener();

                // Register the TrackerListener object to the tracker server.
                tracker.AddTrackerListener(ref listener);

                // Start the tracker server.
                tracker.StartTracker();

                // Perform a system check to see if the tracker server is running OK and print the result.
                Console.Write("System check: " + tracker.Systemcheck() + "\n\n");
                Console.Write("***************************\n\n");

                // Set the frame rate to 30 Hz.
                tracker.SetFramerate(30);
				
                // Print the new frame rate to see if it was set correctly. Note that for PST HD and Pico
                // trackers the frame rate actually being set can differ from the value provided to SetFramerate().
                Console.Write("Frame rate set to " + tracker.GetFramerate() + " Hz\n");

                // Query available exposure range for the current frame rate and try setting the maximum exposure value
                double min = 0.0, max = 0.0;
                tracker.GetExposureRange(ref min, ref max);
                Console.Write("Exposure range: " + min + " s - " + max + " s\n");
                Console.Write("Set exposure to " + max + "\n");
                tracker.SetExposure(max);
                Console.Write("Check new exposure: " + tracker.GetExposure() + " s\n\n");
                Console.Write("***************************\n\n");

                // Increase frame rate and check exposure value. For PST HD and PST Pico trackers, maximum exposure
                // depends on frame rate. Exposure will be automatically decreased when necessary.
                Console.Write("Set frame rate to 60 Hz\n");
                tracker.SetFramerate(60);
                Console.Write("Frame rate set to " + tracker.GetFramerate() + " Hz\n");
                Console.Write("Check exposure: " + tracker.GetExposure() + " s\n");

                // Check new exposure range
                tracker.GetExposureRange(ref min, ref max);
                Console.Write("New exposure range " + min + " s - " + max + " s\n\n");
                Console.Write("***************************\n\n");

                // Set exposure half-way
                double exposureHalf = min + (max - min) / 2.0;
                Console.Write("Set exposure half way: " + exposureHalf + " s\n");
                tracker.SetExposure(exposureHalf);
                Console.Write("New exposure: " + tracker.GetExposure() + " s\n\n");
                Console.Write("***************************\n\n");

                // Main loop, wait for auto-termination.
                while (Running)
                {
                    Thread.Sleep(100);
                }

            }
            catch (Exception e)
            {
                // Catch PSTech.TrackerException exceptions and print error messages.
                Console.Write(e + "\n");
            }
            finally
            {
                // Make sure that the connection to the PST Tracker is shut down properly.
                Tracker.Shutdown();
            }

            // Pause command line to see results.
            Console.Write("Press enter to continue...\n");
            Console.Read();

            return 0;
        }
    }
}
