/**
 * restserver.cpp
 * Copyright PS-Tech B.V. All Rights Reserved.
 *
 * This example shows how to enable the REST server using the PST SDK.
 * The REST server enables network-based access to the PST Tracker using
 * the HTTP protocol. GET and POST requests can be made to the server to
 * send and receive data and change parameters. The REST interface offers
 * a programming language independent interface to the PST Tracker.
 * Besides accessing the REST server directly using a browser, a networking
 * library like Curl can be used to interface with the server programatically.
 * 
 */

#ifdef WIN32
#include <windows.h>
#else
#include <csignal>
#endif

/*
 * Define handler functions required to ensure a clean shutdown of the PST Tracker
 * when the application is terminated.
 */

static void Exithandler(int sig);

#ifdef WIN32
BOOL WINAPI ConsoleHandler(DWORD CEvent)
{
    Exithandler(CEvent);
    return TRUE;
}
#endif

/* End of handler functions */

#include <iostream>
#include <thread>
#include <chrono>

#include "pstsdk_cpp.h"
#include "TrackerExceptions.h"

/*
 * Implement the exit handler to shut-down the PST Tracker connection on application termination.
 */
static void Exithandler(int sig)
{
    PSTech::pstsdk::Tracker::DisableRestServer();
    PSTech::pstsdk::Tracker::Shutdown();
    exit(0);
}

int main(int argc, char *argv[])
{
    // Register the exit handler with the application
#ifdef WIN32
    SetConsoleCtrlHandler((PHANDLER_ROUTINE)ConsoleHandler, TRUE);
#else
    signal(SIGTERM, Exithandler);
    signal(SIGKILL, Exithandler);
    signal(SIGQUIT, Exithandler);
    signal(SIGINT, Exithandler);
#endif

    // Implement error handling of PSTech::TrackerException exceptions to prevent 
    // improper PST Tracker shutdown on errors.
    try
    {
        // Create an instance of the Tracker object using the default configuration path and file names.
#ifdef WIN32
        PSTech::pstsdk::Tracker pst;
#else
        // On Linux, specify the type of grabber that needs to be used as the last parameter: 
        // "basler_ace" for PST HD or "basler_dart" for PST Pico
        PSTech::pstsdk::Tracker pst("","config.cfg","models.db",argv[1]);
#endif

        // Start the tracker server.
        pst.Start();

        // Enable the REST server. To check if the REST server is started correctly, browse to 
        // http://localhost:7278/PSTapi/SystemCheck
        // In order to use 127.0.0.1 an an address on Windows 7, execute the following command 
        // on an elevated command prompt to allow communication to this address:
        // "netsh http add urlacl url=http://127.0.0.1:7278/ user=EVERYONE listen=yes delegate=no"
        pst.EnableRestServer("localhost", "7278");

        std::cout << "PST REST server enabled. See the PST SDK Manual for example commands.\n";

        // Wait for one minute before terminating this application.
        std::this_thread::sleep_for(std::chrono::milliseconds(60000));

        // Disable the REST server
        pst.DisableRestServer();
    }
    catch (PSTech::TrackerException &e)
    {
        std::cout << e.full_description() << "\n";
        // Handle the error.
    }

    // Make sure that the connection to the PST Tracker is shut down properly.
    PSTech::pstsdk::Tracker::Shutdown();

    return 0;
}