"""Module containing all trackerdata related classes and functions.

Copyright PS-Tech B.V. All Rights Reserved.
"""

import ctypes as c

from .point import Point
from .target import TargetPose

class TrackerData:
    """Tracking information retrieved from tracker.

    This class contains all tracking information recorded at a certain time instance. It provides a list of all
    active tracking targets that have been detected by the PST Tracker, as well as a list of all 3D data points
    that have been detected by the tracker and are estimated not to be part of any active tracking target.

    Attributes:
        timestamp: Timestamp of the moment the cameras captured the data. The timestamp uses the system clock provided in seconds since system boot (Windows) or Epoch (Linux).
        seqnumber: Incremental data recording id
        targetlist: target.TargetPose list representing detected tracking targets
        datapoints: point.Point list representing detected 3D markers not matched to a tracking target

    See Also:
        target.TargetPose
        point.Point
        tracker.Tracker.get_single_measurement
        tracker.Listener
    """
    def __init__(self, c_tracker_data):
        self.timestamp = c_tracker_data.timestamp
        self.seqnumber = c_tracker_data.seqnumber

        self.targetlist = []
        for idx in range(c_tracker_data.number_of_targets):
            c_target_pose = c_tracker_data.targetlist[idx]
            target_pose = TargetPose(c_target_pose)
            self.targetlist.append(target_pose)

        self.datapoints = []
        for idx in range(c_tracker_data.number_of_points):
            c_point = c_tracker_data.datapoints[idx]
            point = Point(c_point)
            self.datapoints.append(point)

    def __eq__(self, other):
        if not isinstance(other, TrackerData):
            return NotImplemented
        return  self.timestamp == other.timestamp and \
                self.seqnumber == other.seqnumber and \
                self.targetlist == other.targetlist and \
                self.datapoints == other.datapoints

    ##@cond
    class _CTrackerData(c.Structure):
        _fields_ = [('timestamp', c.c_double),
                    ('seqnumber', c.c_int),
                    ('number_of_targets', c.c_size_t),
                    ('number_of_points', c.c_size_t),
                    ('targetlist', c.POINTER(TargetPose._CTargetPose)),
                    ('datapoints', c.POINTER(Point._CPoint)),
                    ]
    ##@endcond
