"""Module containing all error related classes and functions.

Copyright PS-Tech B.V. All Rights Reserved.
"""
from enum import IntEnum

class TrackerError(Exception):
    """Base tracker exception

    Attributes:
        message: String containing the message of the error.
    """
    def __init__(self, message):
        super().__init__(message)
        self.message = message


class NotInitializedError(TrackerError):
    """A call was made to an uninitialized tracker."""
    pass  # pylint: disable=unnecessary-pass


class NotSupportedError(TrackerError):
    """The requested functionality is not supported by this tracker."""
    pass  # pylint: disable=unnecessary-pass


class OutOfRangeError(TrackerError):
    """The supplied value is outside of the accepted range."""
    pass  # pylint: disable=unnecessary-pass


class NotFoundError(TrackerError):
    """The requested data could not be found."""
    pass  # pylint: disable=unnecessary-pass


class HTTPError(TrackerError):
    """An HTTP error was thrown by the Rest server."""
    pass  # pylint: disable=unnecessary-pass


class FailedToAllocateMemoryError(TrackerError):
    """Failed to allocate memory."""
    pass  # pylint: disable=unnecessary-pass


class NullPointerError(TrackerError):
    """The supplied tracker object was a null pointer."""
    pass  # pylint: disable=unnecessary-pass


class ImageRetrievalError(TrackerError):
    """Image retrieval failed."""
    pass  # pylint: disable=unnecessary-pass


class InvalidDataError(TrackerError):
    """Data provided did not match the requirements."""
    pass  # pylint: disable=unnecessary-pass


class AlreadyExistsError(TrackerError):
    """The request could not be executed because the object alreay exists."""
    pass  # pylint: disable=unnecessary-pass


class JSONError(TrackerError):
    """An error occurred while processing JSON information."""
    pass  # pylint: disable=unnecessary-pass


class EStatusMessage(IntEnum):
    """ Tracker status messages enum class

    This enum class lists all status messages that can be reported by the tracker.

    See Also:
        tracker.Tracker.system_check

    Attributes:
        OK: The system is running OK.
        NOT_INITIALIZED: Tracker failed to initialize correctly.
        DISCONNECTED: The grabber has been disconnected.
        GENERAL_ERROR: Unspecified grabber error.
        TIMEOUT_ERROR: Grabber timeout error.
        NOCAMS_FOUND: Grabber could not detect any cameras.
        NOTENOUGHCAMS_FOUND: Grabber could not detect sufficient cameras.
        INIT_ERROR: Grabber did not initialize correctly.
        CANNOT_START_CAMS: Grabber could not start cameras.
        CANNOT_SETUP_CAMS: Grabber failed setting up cameras.
        NULL_TRACKER_HANDLE_ERROR: Tracker not created or freed.
        UNKNOWN_STATUS: Unknown status message.
    """
    OK                          =  0
    NOT_INITIALIZED             =  1
    DISCONNECTED                =  2
    GENERAL_ERROR               =  3
    TIMEOUT_ERROR               =  4
    NOCAMS_FOUND                =  5
    NOTENOUGHCAMS_FOUND         =  6
    INIT_ERROR                  =  7
    CANNOT_START_CAMS           =  8
    CANNOT_SETUP_CAMS           =  9
    NULL_TRACKER_HANDLE_ERROR   = 10
    UNKNOWN_STATUS              = 11


##@cond
class _EErrorStatus(IntEnum):
    OK                                  = 0
    TRACKER_ERROR                       = 1
    NOT_INITIALIZED_ERROR               = 2
    NOT_SUPPORTED_ERROR                 = 3
    OUT_OF_RANGE_ERROR                  = 4
    NOT_FOUND_ERROR                     = 5
    HTTP_ERROR                          = 6
    FAILED_TO_ALLOCATE_MEMORY_ERROR     = 7
    NULLPTR_ERROR                       = 8
    IMAGE_RETRIEVAL_FAILED_ERROR        = 9
    INVALID_DATA_ERROR                  = 10
    ALREADY_EXISTS_ERROR                = 11
    JSON_ERROR                          = 12

    # pylint: disable=comparison-with-callable
    def raise_error(self, message):
        """Check error status type and raise corresponding error"""
        if self.value == self.TRACKER_ERROR:
            raise TrackerError(message)
        if self.value == self.NOT_INITIALIZED_ERROR:
            raise NotInitializedError(message)
        if self.value == self.NOT_SUPPORTED_ERROR:
            raise NotSupportedError(message)
        if self.value == self.OUT_OF_RANGE_ERROR:
            raise OutOfRangeError(message)
        if self.value == self.NOT_FOUND_ERROR:
            raise NotFoundError(message)
        if self.value == self.HTTP_ERROR:
            raise HTTPError(message)
        if self.value == self.FAILED_TO_ALLOCATE_MEMORY_ERROR:
            raise FailedToAllocateMemoryError(message)
        if self.value == self.NULLPTR_ERROR:
            raise NullPointerError(message)
        if self.value == self.IMAGE_RETRIEVAL_FAILED_ERROR:
            raise ImageRetrievalError(message)
        if self.value == self.INVALID_DATA_ERROR:
            raise InvalidDataError(message)
        if self.value == self.ALREADY_EXISTS_ERROR:
            raise AlreadyExistsError(message)
        if self.value == self.JSON_ERROR:
            raise JSONError(message)
    # pylint: enable=comparison-with-callable
##@endcond
