// Copyright PS-Tech B.V. All Rights Reserved.

#ifndef TRACKER_EXCEPTIONS_H
#define TRACKER_EXCEPTIONS_H

#include "PstBase.h"
#include <stdexcept>

namespace PSTech
{
    /**
     * @brief Generic tracker exception
     *     
     * Base class defining exceptions being thrown by the PST.
     * Specific instances of this class are defined using the
     * DECLARE_PST_EXCEPTION macro.
     *
     * @ingroup pstsdk
     */
    class PST_EXPORT PSTException
    {
    public:
        explicit PSTException(const char* what);
        PSTException(const PSTException& other);
        PSTException(PSTException&& other) noexcept;
        PSTException(const char* function, int line, const char* what);        
        virtual ~PSTException();
        virtual const char* what() const;               /**< Message describing the exception. */
        virtual const char* where() const;              /**< Location in the code where the exception occured. */
        virtual const char* full_description() const;   /**< Returns a string containing the location and message of the error. */
    
    private:
        char* m_what;
        char* m_where;
        char* m_full;
    };

/**
 * Macro for easy definition of specific exceptions inheriting from a base exception.
 */
#define DECLARE_PST_EXCEPTION( child, parent ) \
        class child : public parent \
        { \
        public: \
            explicit child(const char* what) : parent(what) {} \
            child(const char* function, int line, const char* what) : parent(function, line, what) {}\
        }
    
    DECLARE_PST_EXCEPTION(TrackerException, PSTException);            /**< Base tracker exception. */
    DECLARE_PST_EXCEPTION(InvalidDataException, TrackerException);    /**< Data provided did not match the requirements. */
    DECLARE_PST_EXCEPTION(NotFoundException, TrackerException);       /**< The requested data could not be found. */
    DECLARE_PST_EXCEPTION(AlreadyExistsException, TrackerException);  /**< The request could not be executed because the object alreay exists. */
    DECLARE_PST_EXCEPTION(IOException, TrackerException);             /**< Communication error. */
    DECLARE_PST_EXCEPTION(OutOfRangeException, TrackerException);     /**< The supplied value is outside of the accepted range. */
    DECLARE_PST_EXCEPTION(NotSupportedException, TrackerException);   /**< The requested functionality is not supported by this tracker. */
    DECLARE_PST_EXCEPTION(NotInitializedException, TrackerException); /**< A call was made to an uninitialized tracker. */
    DECLARE_PST_EXCEPTION(HttpException, TrackerException);           /**< An HTTP error was thrown by the Rest server. */
    DECLARE_PST_EXCEPTION(JSONException, TrackerException);           /**< An error occurred while processing JSON information. */
}
#endif //TRACKER_EXCEPTIONS_H
