"""Shared memory example of the PST SDK

This example shows how to activate the shared memory communication pipeline
that enables communication of the PST Client application through the PST SDK.
Note that for the shared memory pipeline to function, the application has to
run with elevated access (administrator rights). After enabling shared memory,
the PST Client application can be used to download calibration information and
manage tracking targets.

Copyright PS-Tech B.V. All Rights Reserved.
"""
import context
import time
import sys
import ctypes
import pstech.pstsdk.tracker as pst
import pstech.pstsdk.errors as psterrors

"""Helper function to register the exit handler with the application"""
def register_exit_handler():
    if sys.platform.startswith("linux"):
        import signal
        signal.signal(signal.SIGTERM, exit_handler)
        signal.signal(signal.SIGHUP, exit_handler)
        signal.signal(signal.SIGQUIT, exit_handler)
        signal.signal(signal.SIGINT, exit_handler)
    elif sys.platform.startswith("win"):
        import win32api
        win32api.SetConsoleCtrlHandler(exit_handler, True)

"""Implement the exit handler to shut-down the PST Tracker connection on application termination."""
def exit_handler(*args):
    pst.Tracker.disable_shared_memory()
    pst.Tracker.shutdown()

"""Check if user is an admin."""
def is_admin():
    if sys.platform.startswith("win"):
        try:
            return ctypes.windll.shell32.IsUserAnAdmin()
        except:
            return False
    elif sys.platform.startswith("linux"):
        return True

def main():
	if is_admin():
		if(len(sys.argv) < 2):
			print("\nConfiguration Error: A camera configuration file needs to be specified. This file can be found in the Redist folder of your installation. "
					"See the documentation of the Python bindings for more information.")
			exit(0)

		# Register exit_handler for proper shutdown
		register_exit_handler()

		try:
			# Use Context Manager to prevent improper Tracker shutdown on errors.
			# Create an instance of the Tracker object using the default configuration path and file names.
			with pst.Tracker("", "","", sys.argv[1]) as tracker:
				# Start the tracker server.
				tracker.start()

				# Enable the shared memory pipeline to enable client connections.
				tracker.enable_shared_memory()

				print("Shared memory server initialized. Start the PST Client application to create a connection.\n")

				# Wait for one minute before terminating this application.
				time.sleep(60)

				# Disable the shared memory pipeline. If the PST Client is still connected at this point,
				# it will try to reconnect and otherwise exit.
				tracker.disable_shared_memory()
		except psterrors.TrackerError as err:
			# Catch TrackerError and print error messages.
			print(err.message)
	else:
		# Re-run the program with admin rights
		ctypes.windll.shell32.ShellExecuteW(None, "runas", sys.executable, " ".join(sys.argv), None, 1)

if __name__ == "__main__":
    main()
