/**
 * exposure.cpp
 * Copyright PS-Tech B.V. All Rights Reserved.
 *
 * This example shows how to adjust exposure settings using the PST SDK. It shows how to
 * change exposure settings based on frame rate and the available exposure range for a
 * certain PST Tracker at a certain frame rate.
 *
 */

#ifdef WIN32
#include <windows.h>
#else
#include <csignal>
#endif

/*
 * Define handler functions required to ensure a clean shutdown of the PST Tracker when the
 * application is terminated.
 */

static void Exithandler(int sig);

#ifdef WIN32
BOOL WINAPI ConsoleHandler(DWORD CEvent)
{
    Exithandler(CEvent);
    return TRUE;
}
#endif

/* End of handler functions */

#include <iostream>
#include <thread>
#include <chrono>
#include <atomic>

#include "pstsdk_cpp.h"
#include "TrackerExceptions.h"
#include "PstStringIoStream.h"

/* Control variable for main loop */
static std::atomic<bool> running(true);

/* Number of data points to grab before application termination */
static const uint32_t numberOfSamplesToGrab = 1000;

/*
 * Implementation of the PSTech::pstsdk::Listener class to receive tracking data.
 * The OnTrackerData() callback function receives the data as soon as it becomes
 * available.
 */
class MyListener : public PSTech::pstsdk::Listener
{
    virtual void OnTrackerData(const PSTech::pstsdk::TrackerData& td)
    {
        static uint32_t samplesGrabbed = 0;
        if (samplesGrabbed++ >= numberOfSamplesToGrab)
            running = false;

        // Do something with the received data.
    }
} listener;

/*
 * Implement the exit handler to shut-down the PST Tracker connection on application termination.
 */
static void Exithandler(int sig)
{
    PSTech::pstsdk::Tracker::Shutdown();
    running = false;
}

int main(int argc, char *argv[])
{
    // Register the exit handler with the application
#ifdef WIN32
    SetConsoleCtrlHandler((PHANDLER_ROUTINE)ConsoleHandler, TRUE);
#else
    signal(SIGTERM, Exithandler);
    signal(SIGKILL, Exithandler);
    signal(SIGQUIT, Exithandler);
    signal(SIGINT, Exithandler);
#endif

    // Implement error handling of PSTech::TrackerException exceptions to prevent
    // improper PST Tracker shutdown on errors.
    try
    {
        // Create an instance of the Tracker object using the default configuration path and file names.
#ifdef WIN32
        PSTech::pstsdk::Tracker pst;
#else
        // On Linux, specify the type of grabber that needs to be used as the last parameter:
        // "basler_ace" for PST HD or "basler_dart" for PST Pico
        PSTech::pstsdk::Tracker pst("","config.cfg","models.db",argv[1]);
#endif
        // Print version number of the tracker server being used.
        std::cout << "Running PST Server version " << pst.GetVersionInfo() << "\n";

        // Register the listener object to the tracker server.
        pst.AddTrackerListener(&listener);

        // Start the tracker server.
        pst.Start();

        // Perform a system check to see if the tracker server is running OK and print the result.
        std::cout << "System check: " << (int)pst.Systemcheck() << "\n\n";
        std::cout << "***************************\n\n";

        // Set the frame rate to 30 Hz.
        pst.SetFramerate(30);

        // Print the new frame rate to see if it was set correctly. Note that for PST HD and Pico
        // trackers the frame rate actually being set can differ from the value provided to SetFramerate().
        std::cout << "Frame rate set to " << pst.GetFramerate() << " Hz\n";

        // Query available exposure range for the current frame rate and try setting the maximum exposure value
        double min, max;
        pst.GetExposureRange(min, max);
        std::cout << "Exposure range: " << min << " s - " << max << " s\n";
        std::cout << "Set exposure to " << max << "\n";
        pst.SetExposure(max);
        std::cout << "Check new exposure: " << pst.GetExposure() << " s\n\n";
        std::cout << "***************************\n\n";

        // Increase frame rate and check exposure value. For PST HD and PST Pico trackers, maximum exposure
        // depends on frame rate. Exposure will be automatically decreased when necessary.
        std::cout << "Set frame rate to 60 Hz\n";
        pst.SetFramerate(60);
        std::cout << "Frame rate set to " << pst.GetFramerate() << " Hz\n";
        std::cout << "Check exposure: " << pst.GetExposure() << " s\n";

        // Check new exposure range
        pst.GetExposureRange(min, max);
        std::cout << "New exposure range " << min << " s - " << max << " s\n\n";
        std::cout << "***************************\n\n";

        // Set exposure half-way
        double exposureHalf = min + (max - min) / 2.0;
        std::cout << "Set exposure half way: " << exposureHalf << " s\n";
        pst.SetExposure(exposureHalf);
        std::cout << "New exposure: " << pst.GetExposure() << " s\n\n";
        std::cout << "***************************\n\n";

        // Main loop, wait for auto-termination.
        while (running)
        {
            std::this_thread::sleep_for(std::chrono::milliseconds(100));
        }
    }
    catch (PSTech::TrackerException &e)
    {
        // Catch PSTech::TrackerException exceptions and print error messages.
        std::cout << e.full_description() << "\n";
    }

    // Make sure that the connection to the PST Tracker is shut down properly.
    PSTech::pstsdk::Tracker::Shutdown();

    // Pause command line to see results.
    std::cout << "Press enter to continue...\n";
    std::cin.get();

    return 0;
}