// Copyright PS-Tech B.V. All Rights Reserved.

#pragma once

#include "PstBase.h"
#include "PstString.h"
#include "PstVector.h"
#include "PstArray.h"
#include "PstTypes.h"

namespace PSTech
{
namespace pstsdk
{
    /**
    * @defgroup pstsdkcpp PST SDK C++ API
    * This library defines the C++ API for the PST. Applications implementing this API
    * get direct access to the PST to receive tracking data and set tracker parameters.
    * Using this library, it is also possible to initialize a shared memory communication
    * pipeline to connect to the PST Client application and initialize a REST server for
    * HTTP-based communication.
    * @{
    */
   
    /**
     * @brief Abstract listener class for receiving tracking information and tracking mode updates.
     *
     * Extend this class and implement its methods in order to receive tracking information and tracking mode updates from the PST Tracker.
     * Alternatively, Tracker::GetSingleMeasurement() can be called to get the latest available tracking data.
     * 
     * @see Tracker::AddTrackerListener()
     * @see Tracker::RemoveTrackerListener()
     * @see Tracker::GetSingleMeasurement()
     */
    class Listener
    {
    public:
        /**
         * @brief Callback function receiving tracking information from the tracker.
         *
         * Implement the OnTrackerData method to receive tracking information from the PST Tracker.
         *
         * @param[out] data A TrackerData object containing the detected tracking targets and 3D markers.
         *
         * @see TrackerData
         */
        virtual void OnTrackerData(const TrackerData& data) { }
        
        /**
         * @brief Callback function receiving tracking mode updates from the the tracker.
         *
         * Implement the OnTrackerMode method to receive tracking mode updates of the PST Tracker.
         * This function enables responding to the different PST Tracker mode changes enumerated by TrackerMode as soon as they occur.
         *
         * @param[out] mode Current tracker mode code as TrackerMode enumeration member.
         *
         * @see TrackerMode
         */
        virtual void OnTrackerMode(const TrackerMode mode) { }
        virtual ~Listener() { }
    };

    /**
     * @brief Write plug-in status information to the standard C output stream stderr.
     * 
     * Provides information about the status of the tracker plug-in. Information is written to the standard C output stream stderr using a std::clog stream. Useful when debugging.
     * 
     * @note This should be called before creating any Tracker objects. Calling this afterwards will have no effect.
     */
    PST_EXPORT void EnableLogging();

    /**
     * @brief Retrieve the SDK version string.
     * 
     * Retrieves the SDK version information as a c style string. The SDK version can differ from the internal tracker version which can be retrieved using Tracker::GetVersionInfo().
     * 
     * @see Tracker::GetVersionInfo()
     */
    PST_EXPORT const char* GetSdkVersion();

    /**
     * @brief Main PST SDK class implementing tracker communication.
     *
     * This class implements the main PST SDK functionality. Examples on how to use this class and how to get data from the PST Tracker
     * can be found in the Examples section.
     */
    class PST_EXPORT Tracker
    {
    public:
        /**
         * @brief Tracker constructor.
         *
         * Construct Tracker object and initialize connected PST tracker. Camera calibration information is checked during initialization.
         * When calibration files for the connected tracker can not be found, a message with download links will be printed to the command line.
         *
         * @param[in] path Path to the configuration directory to be used for storing camera calibrations, target database and configuration files. Use "" for default. (default: "%PROGRAMDATA%\PS-Tech\PST Iris" on Windows or ~/.pstiris on Linux)
         * @param[in] config_file Name of the server configuration file. Use "" for default. (default: "server.cfg")
         * @param[in] db_file Name of the tracking target database file. Use "" for default. (default: "models.db")
         * @param[in] grabber_name Name of the grabber plugin to be used. Can contain absolute or relative path to xml config file. Set to "default" to use auto-detect. (default: "default")
         *
         * @attention 
         * Before destroying a Tracker object the static Tracker::Shutdown() function must always be called to ensure proper shutdown.
         * 
         * @see Tracker::GetUncalibratedCameraUrls()
         * @see Tracker::Shutdown()
         *
         * @exception PSTech::TrackerException Unable to initialize Tracker
         * @exception PSTech::NotInitializedException Tracker was not initialized correctly
         */
        Tracker(const Utils::PstString& path = "", const Utils::PstString& config_file = "server.cfg",
            const Utils::PstString& db_file = "models.db", const Utils::PstString& grabber_name = "default");
        ~Tracker();

        /**
         * @brief Load calibration information from a given path.
         *
         * Load the calibration information for each of the cameras in the connected PST Tracker from the specified local path. This function will overwrite 
         * any calibration information for the currently connected trackers available in the current configuration folder. Calibration information will only 
         * be loaded and placed in the configuration folder when calibration information is found and valid for all connected cameras.
         * 
         * @param[in] path Path to the local directory where the calibration information for all connected cameras is stored.
         *
         * @see Tracker::GetConnectedCameraUrls()
         *
         * @exception PSTech::NotFoundException Not all the required files could be found in the specified location.
         * @exception PSTech::InvalidDataException Calibration file contains invalid data.
         */
        void LoadCalibrationFromLocalPath(const Utils::PstString& path);

        /**
         * @brief Retrieve the latest TrackerData available from the connected PST Tracker.
         *
         * Retrieve the latest TrackerData available from the connected PST Tracker. If this function is called before the tracking system is started with
         * Tracker::Start(), the resulting TrackerData object will be empty. 
         * 
         * @return The latest TrackerData retrieved from the PST Tracker.
         * 
         * @see TrackerData
         * @see Tracker::Start()
         * @see Listener 
         */
        TrackerData GetSingleMeasurement() const;

        /**
         * @brief Get URLs of the calibration information for the connected PST Tracker.
         *
         * Returns and prints the URLs to download the calibration information for the cameras in the connected PST Tracker.
         *
         * @param[in] silent Set to true in order to disable printing of calibration information. (default: false)
         * @return Vector of URLs specifying the download location of the calibration information. The size of the vector is the number of connected cameras.
         */
        Utils::PstVector<Utils::PstString> GetConnectedCameraUrls(bool silent = false) const;

        /**
         * @brief Check if cameras of the connected PST Tracker are calibrated.
         *
         * Check for calibration information of the cameras of the connected PST Tracker. When not all calibration information can be retrieved,
         * download urls for the uncalibrated cameras will be returned. Furthermore, a message will be shown at the command line specifying which
         * files are missing and where they can be downloaded.
         *
         * @param[in] silent Set to true in order to disable printing of missing calibration information. (default: false)
         * @return Vector of URLs specifying the download location of calibration information for uncalibrated cameras. The size of the vector is the number of uncalibrated cameras.
         */
        Utils::PstVector<Utils::PstString> GetUncalibratedCameraUrls(bool silent = false);

        /**
         * @brief Get version information of the SDK.
         *
         * Retrieves version information of the PST Server used by the SDK. This version can differ from the SDK version retrieved using GetSdkVersion().
         *
         * @return The version number of the PST Server.
         * 
         * @see GetSdkVersion()
         */
        Utils::PstString GetVersionInfo() const;

        /**
         * @brief Get the path to the current configuration directory.
         *
         * Retrieves the full path to the configuration directory currently being used by the PST Tracker. The default path is "%PROGRAMDATA%\PS-Tech\PST Iris" on Windows or ~/.pstiris on Linux.
         * The configuration directory contains the target model database, server configuration, reference file and tracker calibration information.
         *
         * @return The full path to the configuration directory.
         */
        Utils::PstString GetConfigPath() const;


        /**
         * @brief Add a listener for receiving tracker data and tracking mode updates.
         *
         * Register a listener to receive tracking information and tracking mode updates from the connected PST Tracker. The use of this method requires an implementation of the Listener class to be made.
         *
         * @param[in] listener A pointer to an implementation of the Listener class to function as a callback.
         *
         * @see Listener
         * @see TrackerData
         * @see TrackerMode
         * @see Tracker::RemoveTrackerListener()
         */
        void AddTrackerListener(Listener* listener);

        /**
         * @brief Remove a listener for receiving tracker data and tracking mode updates.
         *
         * Remove a listener receiving tracking information and tracking mode updates from the connected PST Tracker.
         *
         * @param[in] listener A pointer to an implementation of the Listener class which has previously been added using Tracker::AddTrackerListener().
         *
         * @see Listener
         * @see Tracker::AddTrackerListener()
         */
        void RemoveTrackerListener(Listener* listener);

        /**
         * @brief Start tracking.
         *
         * Start the tracking system and produce tracking results.
         *
         * @see Tracker::Pause()
         */
        void Start();

        /**
         * @brief Pause tracking
         *
         * Pause the tracking system, stop producing tracking results and enter low-power mode.
         * Tracking can be resumed with a subsequent call to Tracker::Start().
         *
         * @see Tracker::Start()
         */
        void Pause();

        /**
         * @brief Shutdown the tracking system, stopping tracking.
         *
         * Fully shutdown the tracking system, disconnecting from the PST Tracker and rendering the Tracker object useless.
         * When the shared memory interface or the REST server have been enabled these will be disabled as well.
         * In order to restart the tracker, a new instance of the Tracker class has to be created.
         * 
         * @see Tracker::Tracker()
         * @see Tracker::DisableSharedMemory()
         * @see Tracker::DisableRestServer()
         */
        static void Shutdown();

        /**
         * @brief Check if the tracker is running correctly.
         *
         * Perform a system check, checking if the PST Tracker has been initialized correctly and if the system is running properly.
         * In order to get continuous feedback on the status of the PST Tracker it is recommended to implement regular polling of this
         * function. This way, issues that will not cause the software to crash (e.g. a camera disconnect) can be handled.
         *
         * @return Current system status code as StatusMessage enumeration member.
         *
         * @see StatusMessage
         */
        StatusMessage Systemcheck() const;


        /**
         * @brief Set tracker frame rate.
         *
         * Set the PST Tracker frame rate to the value available for the connected PST Tracker nearest to the supplied value.
         *
         * @param[in] fps The new frame rate to be set.
         *
         * @see Tracker::GetFramerate()
         * @see Tracker::GetSupportedFramerates()
         *
         * @exception PSTech::NotSupportedException Supplied frame rate could not be set on the Tracker.
         */
        void SetFramerate(double fps);

        /**
         * @brief Get current frame rate.
         *
         * Get the current frame rate as set on the PST Tracker.
         *
         * @return Current frame rate in frames per second (fps).
         *
         * @see Tracker::SetFramerate()
         * @see Tracker::GetSupportedFramerates()
         */
        double GetFramerate() const;

        /**
         * @brief Get vector of available frame rates.
         *
         * Get a vector containing all frame rates that are available for the currently connected PST Tracker.
         * The range of available frame rates differs for different types of PST Trackers. When using
         * Tracker::SetFramerate() to set a new frame rate, the frame rate of the PST Tracker will be set to the
         * value provided by this function nearest to the value supplied to Tracker::SetFramerate().
         *
         * @return A vector of available frame rates.
         *
         * @see Tracker::SetFramerate()
         * @see Tracker::GetFramerate()
         */
        Utils::PstVector<float> GetSupportedFramerates() const;

        /**
         * @brief Set the exposure time.
         *
         * Set the PST Tracker exposure time to the supplied time. This adjusts both the shutter speed of the PST Tracker and the time the
         * IR illumination panel will be turned on.
         *
         * @param[in] time Exposure time in seconds.
         *
         * @see Tracker::GetExposure()
         * @see Tracker::GetExposureRange()
         *
         * @exception PSTech::OutOfRangeException The value supplied is not within the allowed range.
         * @exception PSTech::NotSupportedException Exposure could not be set on the Tracker.
         */
        void SetExposure(double time);

        /**
         * @brief Get the current exposure time.
         *
         * Get the current exposure time as set on the PST Tracker.
         *
         * @return The current exposure time in seconds.
         *
         * @see Tracker::SetExposure()
         */
        double GetExposure() const;

        /**
         * @brief Get the allowed exposure range.
         *
         * Get the minimum and maximum exposure value that can be set with the current frame rate.
         * The maximum exposure value differs for different frame rates and PST Trackers.
         * In general, lower frame rates allow for higher exposures and higher frame rates have a
         * lower maximum exposure value. After changing the frame rate it is advised to check the
         * currently allowed range before changing exposure values.
         *
         * @param[out] min The minimum exposure value that can be set.
         * @param[out] max The maximum exposure value that can be set.
         *
         * @see Tracker::GetExposure()
         * @see Tracker::SetExposure()
         * @see Tracker::SetFramerate()
         */
        void GetExposureRange(double& min, double& max) const;

        /**
         * @brief Enable filtering of the tracking results.
         *
         * Switch on filtering of the tracker results using a double exponential-based prediction filter (DESP).
         * This will result in smoother tracking with less noise. It can have a small impact on tracker accuracy.
         *
         * @note When the shared memory communication pipeline is used to connect the PST-Client application to the
         * SDK, filtering will be disabled upon starting the PST-Client application. To prevent multiple filter passes,
         * filtering should not be re-enabled until the PST-Client application is closed.
         *
         * @see Tracker::DisableFiltering()
         * @see Tracker::SetPositionFilter()
         * @see Tracker::SetOrientationFilter()
         * @see Tracker::EnableTremorFilter()
         * @see Tracker::EnableSharedMemory()
         */
        void EnableFiltering();

        /**
         * @brief Disable filtering of the tracking results.
         *
         * Switch off filtering of the tracking results using the double exponential-based prediction filter (DESP).
         *
         * @see Tracker::EnableFiltering()
         * @see Tracker::DisableTremorFilter()
         */
        void DisableFiltering();

        /**
         * @brief Set the strength of the position filter.
         *
         * Set the strength of the double exponential-based prediction filter (DESP) that filters the estimated positions of the tracked tracking targets.
         * The supplied filtering value should be between 0 (no filtering) and 0.99 (maximum filtering).
         *
         * @param[in] value Filtering strength to be used in range [0, 0.99]
         *
         * @see Tracker::EnableFiltering()
         * @see Tracker::SetOrientationFilter()
         */
        void SetPositionFilter(double value);

        /**
         * @brief Set the strength of the orientation filter.
         *
         * Set the strength of the double exponential-based prediction filter (DESP) that filters the estimated orientation of the tracked tracking targets.
         * The supplied filtering value should be between 0 (no filtering) and 0.99 (maximum filtering).
         *
         * @param[in] value Filtering strength to be used in range [0, 0.99]
         *
         * @see Tracker::EnableFiltering()
         * @see Tracker::SetPositionFilter()
         */
        void SetOrientationFilter(double value);

        /**
         * @brief Enable the tremor filter.
         *
         * Enables the tremor filter. This filter greatly reduces the noise levels of the estimated target pose. However, using the filter introduces latency
         * to the tracking results received and lowers tracking accuracy. This filter is mainly useful when using the PST Tracker in an interaction-type setting.
         * When the PST Tracker is used for measurement purposes, enabling the tremor filter is not recommended.
         *
         * @note When the shared memory communication pipeline is used to connect the PST-Client application to the
         * SDK, filtering will be disabled upon starting the PST-Client application. To prevent multiple filter passes,
         * filtering should not be re-enabled until the PST-Client application is closed.
         *
         * @see Tracker::DisableTremorFilter()
         * @see Tracker::EnableFiltering()
         */
        void EnableTremorFilter();

        /**
         * @brief Disable the tremor filter.
         *
         * Disables the tremor filter. The tremor filter will no longer be used to filter the estimated target pose.
         *
         * @see Tracker::EnableTremorFilter()
         * @see Tracker::DisableFiltering()
         */
        void DisableTremorFilter();


        /**
         * @brief Enable image transfer from the PST Tracker.
         *
         * Enable transferring grayscale image data from the PST Tracker. After enabling image transfer, call Tracker::GetImage()
         * in order to receive the captured images. When no calls are made to Tracker::GetImage() for more than 4 seconds, image
         * transfer will be automatically disabled. Note that for the standard PST Iris and standard PST Base trackers, enabling image
         * transfer will result in a reduced framerate of 30 fps. The frame rate will be restored to the original setting after
         * image transfer is disabled.
         *
         * @see Tracker::GetImage()
         * @see Tracker::DisableImageTransfer()
         */
        void EnableImageTransfer();

        /**
         * @brief Disable image transfer from the PST Tracker.
         *
         * Disable transferring grayscale image data from the PST Tracker. When using a standard PST Iris or PST Base tracker,
         * the frame rate of the tracker will be restored to the frame rate that was set before enabling image transfer.
         *
         * @see Tracker::EnableImageTransfer()
         */
        void DisableImageTransfer();

        /**
         * @brief Retrieve images from the connected PST Tracker.
         *
         * Retrieve images from the PST Tracker when image transfer has been enabled. When image transfer has not been enabled by
         * calling Tracker::EnableImageTransfer() before calling this method, Image.image will be an empty vector. Note that
         * image transfer will be disabled automatically when no call to this method is made for 4 seconds.
         *
         * @param[out] image The image information retrieved from the PST Tracker.
         * @return Success of the operation as a boolean.
         *
         * @see Image
         * @see Tracker::EnableImageTransfer()
         */
        bool GetImage(Image& image) const;


        /**
         * @brief Get TargetStatuses object containing all tracking targets and their status.
         *
         * Retrieves the list of all tracking targets defined in the tracking target database and returns them as a TargetStatuses object.
         *
         * @return TargetStatuses object containing the information and status of all available tracking Targets.
         *
         * @see TargetStatus
         * @see TargetStatuses
         * @see Tracker::SetTargetStatus()
         * @see Tracker::GetTargetStatus()
         */
        TargetStatuses GetTargetList() const;

        /**
         * @brief Get basic tracking target information.
         *
         * Retrieves the name, uuid and id for the requested tracking target.
         *
         * @param[in] name The name of the tracking target.
         * @return Target object containing the basic target information.
         *
         * @see Target
         *
         * @exception PSTech::NotFoundException Target name could not be found.
         */
        Target GetTargetInfo(const Utils::PstString& name) const;

        /**
         * @brief Set status of a single tracking Target.
         *
         * Sets the status of a single tracking Target to active (true) or inactive (false).
         *
         * @param[in] name The name of the tracking Target to set the status of.
         * @param[in] active New status of the tracking Target.
         *
         * @see TargetStatus
         * @see Tracker::GetTargetStatus()
         * @see Tracker::GetTargetList()
         *
         * @exception PSTech::NotFoundException Target name could not be found.
         */
        void SetTargetStatus(const Utils::PstString& name, bool active);

        /**
         * @brief Get status of a single tracking Target.
         *
         * Retrieves the status of a single tracking Target. Tracking targets can be either active (true) or inactive (false).
         *
         * @param[in] name The name of the tracking Target to get the status of.
         * @return Status of the tracking Target.
         *
         * @see TargetStatus
         * @see Tracker::SetTargetStatus()
         * @see Tracker::GetTargetList()
         *
         * @exception PSTech::NotFoundException Target name could not be found.
         */
        bool GetTargetStatus(const Utils::PstString& name) const;

        /**
         * @brief Get 3D marker positions of stored tracking Target.
         *
         * Gets a list of the 3D positions of the markers making up the specified tracking Target. The returned positions are
         * relative to the tracking Target origin.
         *
         * @param[in] name Name of the tracking Target to get the marker positions of.
         * @return A list of 3D marker positions.
         *
         * @see MarkerList
         * @see Target
         *
         * @exception PSTech::NotFoundException Target name could not be found.
         */
        MarkerList GetTargetMarkers(const Utils::PstString& name) const;

        /**
         * @brief Set the id of the tracking target.
         *
         * This function changes the id of the tracking target. This id is mainly used when connecting to the PST Tracker through VRPN.
         *
         * @param[in] name The name of the tracking target to change the id of.
         * @param[in] id The new id of the tracking target.
         *
         * @see Target
         *
         * @exception PSTech::NotFoundException Target name could not be found.
         */
        void SetTargetId(const Utils::PstString& name, int id);

        /**
         * @brief Set the reference system in which tracking results are reported
         *
         * Sets the reference system relative to which tracking results are reported. The reference system is defined by a homogeneous transformation matrix
         * (a row-major 4x4 matrix) representing a right-handed Cartesian coordinate system. The function checks if the rotation component of the supplied 
         * transformation matrix is orthonormal and if the supplied transformation matrix is homogeneous (i.e. the final row of the matrix is [0 0 0 1]). If 
         * either one of these conditions is not true, a PSTech::TrackerException is thrown. As an optional parameter, the reference system can be set 
         * relative to the current reference system. When this option is used, the current reference system is transformed using the supplied transformation 
         * matrix. Otherwise, the current reference system is replaced by the supplied reference system.
         *
         * @param[in] reference The transformation matrix for the new reference system.
         * @param[in] relative When set to true, the current transformation matrix is transformed using the supplied transformation matrix.
         *
         * @see Tracker::GetReference()
         * @see Tracker::SetDefaultReference()
         * 
         * @exception PSTech::TrackerException The supplied reference matrix is either non-homogeneous or the rotation component is non-orthonormal.
         */
        void SetReference(const Utils::PstArray<float, 16>& reference, bool relative = false);

        /**
         * @brief Reset the reference system to the default reference system.
         *
         * Resets the currently set reference system to the default reference system. The default system is defined by a 4x4 identity matrix.
         * This places the origin of the reference system 1 meter away from the PST Tracker. It is oriented such that the Z-axis points away
         * from the PST tracker and the X-axis is parallel to the PST tracker.
         *
         * @see Tracker::SetReference()
         * @see Tracker::GetReference()
         */
        void SetDefaultReference();

        /**
         * @brief Gets the transformation matrix for the current reference system.
         *
         * Tracking results are reported relative to a predefined right-handed Cartesian coordinate system, called the reference system.
         * The default reference system is located at 1 meter from the PST Tracker. It is oriented such that the Z-axis points away from
         * the PST tracker and the X-axis is parallel to the PST tracker. The transformation matrix defining the reference system is a
         * row-major 4x4 homogeneous transformation matrix.
         *
         * @return The transformation matrix for the current reference system.
         *
         * @see Tracker::SetReference()
         * @see Tracker::SetDefaultReference()
         */
        Utils::PstArray<float, 16> GetReference() const;

        /**
         * @brief Enable shared memory communication layer.
         *
         * Enable a shared memory-based communication layer. Enabling shared memory will allow the PST-Client application to connect to the SDK.
         * This will allow receiving tracking data and setting parameters using the PST-Client application.
         * Note that a PST-Client that is connected in this way will not be updated when settings are changed through the SDK. This could cause
         * instabilities in the PST software. It is recommended to only use this function when setting up a PST Tracker (e.g. adding tracking
         * targets or retrieving calibration data). When using a tracker in normal operation, it is recommended to choose to use either the
         * PST-Client application or the SDK and not mix usage.
         *
         * @note When using shared memory on Windows the application needs to have elevated access (run as Administrator). When running without
         * elevated access the PST-Client will not be able to connect.
         *
         * @note When the shared memory communication pipeline is used to connect the PST-Client application to the
         * SDK, filtering will be disabled upon starting the PST-Client application. To prevent multiple filter passes,
         * filtering should not be re-enabled until the PST-Client application is closed.
         * @note A Tracker object must have been initialized before enabling shared memory.
         *
         * @see Tracker::DisableSharedMemory()
         * @see Tracker::EnableFiltering()
         * @see Tracker::EnableTremorFilter()
         * 
         * @exception PSTech::NotInitializedException A Tracker object must have been initialized before enabling shared memory.
         *
         */
        static void EnableSharedMemory();

        /**
         * @brief Disable shared memory communication layer.
         *
         * Disable the shared memory-based communication layer. The PST-Client application will no longer be able to connect to the PST Tracker.
         *
         * @see Tracker::EnableSharedMemory()
         */
        static void DisableSharedMemory();

        /**
         * @brief Enable a REST Server using the HTTP protocol on a local area network.
         *
         * Enable a REST Server that uses the HTTP protocol to enable access to the PST Tracker over a local area network. 
         * Parameters from the tracker can be accessed using GET requests. Tracking results and image can be streamed using an Event-stream based communication pipeline.
         * The state of the tracker and parameters can be set through POST calls.
         * If a REST server is aleady running, calling Tracker::EnableRestServer() will terminate the current server and restart it with the new parameters.
         * More information on how to use the REST server can be found in the \ref REST documentation.
         *
         * @note A Tracker object must have been initialized before enabling the REST server.
         *
         * @param[in] server_address The IP address on which the REST server should be accessible.
         * @param[in] server_port The port number on which the REST server should be accessible.
         * @param[in] eventStream_retry_timeout_ms Timeout in milliseconds before an eventstream attempt to reconnect to the server automatically.
         *
         * @see Tracker::DisableRestServer()
         *
         * @exception PSTech::NotInitializedException A Tracker object must have been initialized before enabling the REST server.
         * @exception PSTech::HttpException The REST server failed to initialize properly. A possible cause could be an invalid IP address or port number.
         */
        static void EnableRestServer(const Utils::PstString& server_address, const Utils::PstString& server_port, int eventStream_retry_timeout_ms = 3000);

        /**
         * @brief Disable the REST server communication layer.
         *
         * Disable the REST server communication layer. If a REST server is running, the connections will be closed and the REST server will be destroyed.
         * If no REST server is running a call to Tracker::DisableRestServer() does nothing.
         *
         * @see Tracker::EnableRestServer()
         */
        static void DisableRestServer();

        /**
         * @brief Import a tracking target model into the target model database.
         *
         * Import a JSON-formatted string representing a tracking target model into the target model database.
         * The JSON string should be formatted according to the instructions found in Appendix A of the PST Manual.
         * After successfully importing the model it can be tracked by the PST Tracker. 
         *
         * @param[in] json_model A JSON-formatted string representing the tracking target model.
         *
         * @see Tracker::ExportJsonModel()
         *
         * @exception PSTech::InvalidDataException Data provided did not match the requirements.
         * @exception PSTech::AlreadyExistsException The request could not be executed because a model with the same name already exists in the database.
         * @exception PSTech::JSONException The JSON string could not be parsed correctly.
         */
        void ImportJSONModel(const Utils::PstString& json_model);

        /**
         * @brief Export a tracking target model as a JSON-formatted string.
         *
         * Export a tracking target model from the target model database as a JSON-formatted string.
         * The string will be formatted according to the specification found in Appendix A of the PST Manual.
         *
         * @param[in] model_name The name of the target model to be exported.
         * 
         * @return A JSON-formatted string representing the requested tracking target model.
         *
         * @see Tracker::ImportJsonModel()
         * 
         * @exception PSTech::NotFoundException The requested model was not found in the database.
         */
        Utils::PstString ExportJSONModel(const Utils::PstString& model_name) const;

        /**
         * @brief Remove a tracking target model from the target model database.
         *
         * Remove a tracking target model from the target model database.
         *
         * @param[in] model_name The name of the tracking target model to be removed.
         *
         * @see Tracker::ImportJsonModel()
         * @see Tracker::ExportJsonModel()
         *
         * @exception PSTech::NotFoundException The tracking target model to be removed was not found in the target model database.
         */
        void RemoveTargetModel(const Utils::PstString& model_name);
    };
    /** @} */
}
}
