// Copyright PS-Tech B.V. All Rights Reserved.

#pragma once

#include "PstBase.h"

#include <cstddef>

namespace PSTech
{
namespace Utils
{
    /**
     * Basic string class, cloning std::string.
     * This class can safely be passed across the DLL boundary.
     */
    class PST_EXPORT PstString
    {
    public:
        /** Default constructor creating an empty PstString. */
        PstString();

        /** Converting constructor converting c-string `string' to PstString. */
        PstString(const char* string);

        /** Copy constructor. */
        PstString(const PstString& string);

        /** Move constructor. */
        PstString(PstString&& string) noexcept;

        /** Destructor */
        ~PstString();

        /** Returns a non-modifyable standard C character array version of the PstString. */
        const char* c_str() const;

        /**
         * Returns the number of characters in the PstString.
         * Equivalent to PstString::size()
         * @see PstString::size()
         */
        size_t length() const;
        
        /**
         * Returns the number of characters in the PstString.
         * Equivalent to PstString::length()
         * @see PstString::length()
         */
        size_t size() const;

        /** Changes the number of characters stored in the PstString to `size' characters. */
        void resize(size_t size);

        /** 
         * Compares this PstString to `string'.
         * 
         * - Returns 0 if the contents of the two PstStrings is equal.
         * - Returns a negative value if this PstString appears before `string', in lexicographical order.
         * - Returns a positive value if this PstString appears after `string', in lexicographical order.
         */
        int compare(const PstString& string) const;

        /**
         * Concatenates the PstString supplied by `string' to this PstString.
         * @return A reference to this concatenated PstString.
         */
        PstString& append(const PstString& string);
        
        /** Destroy the contents of the PstString, leaving it empty. */
        void clear();

        /** Returns true if this PstString contains no characters. */
        bool empty() const;

        /** Assign values to the PstString by copying. */
        PstString& operator=(const PstString& string);

        /** Assign values to the PstString by moving. */
        PstString& operator=(PstString&& string) noexcept;

        /**
         * Lexicographical comparison of two PstStrings.
         * @return True when PstStrings are equal.
         */
        bool operator==(const PstString& string) const;
        
        /**
         * Lexicographical comparison of two PstStrings.
         * @return True when PstStrings are not equal.
         */
        bool operator!=(const PstString& string) const;

        /**
         * Lexicographical comparison of two PstStrings.
         * @return True when this PstString appears before `string'.
         */
        bool operator<(const PstString& string) const;

        /**
         * Lexicographical comparison of two PstStrings.
         * @return True when this PstString appears before or is equal to `string'.
         */
        bool operator<=(const PstString& string) const;

        /**
         * Lexicographical comparison of two PstStrings.
         * @return True when this PstString appears after `string'.
         */
        bool operator>(const PstString& string) const;

        /**
         * Lexicographical comparison of two PstStrings.
         * @return True when this PstString appears after or is equal to `string'.
         */
        bool operator>=(const PstString& string) const;

        /**
         * Appends the string supplied by `right' to this PstString.
         * @return A reference to this appended PstString.
         * @see PstString::append()
         */
        PstString& operator+=(const PstString& right);

        /**
         * Concatenate the PstString supplied by `right' to the PstString supplied by `left'.         
         * @return The concatenation of the two PstStrings.
         * @see PstString::append()
         */
        PST_EXPORT friend PstString operator+(PstString left, const PstString& right);

    private:
        char* m_string;
        size_t m_size;

        void copy_internal(const PstString& string);

        void move_internal(PstString&& string) noexcept;
    };
}
}
