// Copyright PS-Tech B.V. All Rights Reserved.

#pragma once

#include "TrackerExceptions.h"

#include <cstddef>
#include <cassert>
#include <string>

namespace PSTech
{
namespace Utils
{
    /**
     * Basic array class, cloning std::array<T, Size>.
     * This class can safely be passed across the DLL boundary.
     */
    template<typename T, size_t Size>
    class PstArray
    {
    public:
        typedef T value_type;
        typedef size_t size_type;
        typedef T* pointer;
        typedef const T* const_pointer;
        typedef T& reference;
        typedef const T& const_reference;
        typedef T* iterator;
        typedef const T* const_iterator;

        /**
         * Get the number of elements in this PstArray.
         */
        constexpr size_type size() const
        {
            return Size;
        }
        
        /** Returns true if this PstArray is empty. */
        constexpr bool empty() const
        {
            return Size == 0;
        }

        /** Direct access to the underlying array.  */
        const_pointer data() const
        {
            return _array;
        }

        /** Direct access to the underlying array. */
        pointer data()
        {
            return _array;
        }

        /**
         * Get a const reference to the item stored at position `index' in the PstArray with bounds check.
         * If `index' is larger than the size of the PstArray, a PSTech::OutOfRangeException is thrown.
         * @exception PSTech::OutOfRangeException
         */
        const_reference at(size_type index) const
        {
            check_bounds(index);
            return _array[index];
        }

        /**
         * Get a reference to the item stored at position `index' in the PstArray with bounds check.
         * If `index' is larger than the size of the PstArray, a PSTech::OutOfRangeException is thrown.
         * @exception PSTech::OutOfRangeException
         */
        reference at(size_type index)
        {
            check_bounds(index);
            return _array[index];
        }

        /** Get a const reference to the first element in the PstArray. */
        const_reference front() const
        {
            assert(Size > 0);
            return _array[0];
        }

        /** Get a reference to the first element in the PstArray. */
        reference front()
        {
            assert(Size > 0);
            return _array[0];
        }

        /** Get a const reference to the last element in the PstArray. */
        const_reference back() const
        {
            assert(Size > 0);
            return _array[Size - 1];
        }

        /** Get a reference to the last element in the PstArray. */
        reference back()
        {
            assert(Size > 0);
            return _array[Size - 1];
        }

        /** Get a const iterator to the beginning of the PstArray. */
        const_iterator cbegin() const
        {
            return _array;
        }

        /** Get a const iterator to the beginning of the PstArray. */
        const_iterator begin() const
        {
            return _array;
        }

        /** Get an iterator to the beginning of the PstArray. */
        iterator begin()
        {
            return _array;
        }

        /** Get a const iterator to the end of the PstArray. */
        const_iterator cend() const
        {
            return _array + Size;
        }

        /** Get a const iterator to the end of the PstArray. */
        const_iterator end() const
        {
            return _array + Size;
        }

        /** Get an iterator to the end of the PstArray. */
        iterator end()
        {
            return _array + Size;
        }

        /** Get a const reference to the item stored at position `index' in the PstArray. */
        const_reference operator [] (size_type index) const
        {
            assert(index < Size);
            return _array[index];
        }

        /** Get a reference to the item stored at position `index' in the PstArray. */
        reference operator [] (size_type index)
        {
            assert(index < Size);
            return _array[index];
        }

        /** Returns true if each element in this PstArray is equal to the element at the same index in `array'. */
        bool operator==(const PstArray<T, Size>& array) const
        {
            for (size_t index = 0; index < Size; ++index)
            {
                if (_array[index] != array[index])
                {
                    return false;
                }
            }
            return true;
        }

        /** Returns true if any element in this PstArray is not equal to the element at the same index in `array'. */
        bool operator!=(const PstArray<T, Size>& array) const
        {            
            return !(*this == array);
        }
        
        /** Public member to enable aggregate initialization. Should not be accessed directly. */
        value_type _array[Size];

    private:
        void check_bounds(size_type index)
        {
            if (index >= Size)
            {
                throw OutOfRangeException(("Requested index out of range: " + std::to_string(index) + " >= " + std::to_string(Size)).c_str());
            }
        }
    };
}
}
