"""Exposure example of the PST SDK

This example shows how to adjust exposure settings using the PST SDK. It shows how to
change exposure settings based on frame rate and the available exposure range for a
certain PST Tracker at a certain frame rate.

Copyright PS-Tech B.V. All Rights Reserved.
"""
import context
import time
import sys
import pstech.pstsdk.tracker as pst
import pstech.pstsdk.errors as psterrors

# Control variable for main loop
running = True

# Number of data points to grab before application termination
max_samples = 100

# Global number of samples
samples = 0

"""Implementation of the pst.Listener class to receive tracking data."""
class MyListener(pst.Listener):

    """Implementation of a tracker data callback function

    Implementation of a tracker data callback function. The on_tracker_data
    function receives the data as soon as it becomes available.

    Args:
        tracker_data: Object containing tracking information retrieved from tracker
        status_message: Status message reported by the tracker.

    See Also:
        pstech.pstdk.trackerdata.TrackerData
        pstech.pstsdk.errors.EStatusMessage
    """
    def on_tracker_data(self, tracker_data, status_message):
        global samples
        global running

        if samples >= max_samples:
            running = False

        samples += 1
        # Do something here with the received data

"""Helper function to register the exit handler with the application"""
def register_exit_handler():
    if sys.platform.startswith("linux"):
        import signal
        signal.signal(signal.SIGTERM, exit_handler)
        signal.signal(signal.SIGHUP, exit_handler)
        signal.signal(signal.SIGQUIT, exit_handler)
        signal.signal(signal.SIGINT, exit_handler)
    elif sys.platform.startswith("win"):
        import win32api
        win32api.SetConsoleCtrlHandler(exit_handler, True)


"""Implement the exit handler to shut-down the PST Tracker connection on application termination."""
def exit_handler(*args):
    global running
    pst.Tracker.shutdown()
    running = False
    return True

def main():
    if(len(sys.argv) < 2):
        print("\nConfiguration Error: A camera configuration file needs to be specified. This file can be found in the Redist folder of your installation. "
                "See the documentation of the Python bindings for more information.")
        exit(0)

    # Register exit_handler for proper shutdown
    register_exit_handler()

    try:
        # Use Context Manager to prevent improper Tracker shutdown on errors.
        # Create an instance of the Tracker object using the default configuration path and file names.
        with pst.Tracker("", "","", sys.argv[1]) as tracker:

            # Print version number of the tracker server being used.
            print("Running PST Server version " + tracker.get_version_info())

            # Create listener with callback functions for data and/or mode updates.
            listener = MyListener()
            
            # Register the listener object to the tracker server.
            tracker.add_tracker_listener(listener)

            # Start the tracker server.
            tracker.start()

            # Perform a system check to see if the tracker server is running OK and print the result.
            print("System check: " + str(tracker.system_check()))
            print("***************************\n")

            # Set the frame rate to 30 Hz.
            tracker.set_framerate(30)

            # Print the new frame rate to see if it was set correctly. Note that for PST HD and Pico
            # trackers the frame rate actually being set can differ from the value provided to SetFramerate().
            print("Frame rate set to: " + str(tracker.get_framerate()) + " Hz\n")

            # Query available exposure range for the current frame rate and try setting the maximum exposure value.
            min, max = tracker.get_exposure_range()
            print("Exposure range: " + str(min) + " s - " + str(max) + " s")
            print("Set Exposure to " + str(max))
            tracker.set_exposure(max)
            print("Check new exposure: " + str(tracker.get_exposure()) + " s")
            print("***************************\n")


            # Increase frame rate and check exposure value. For PST HD and PST Pico trackers, maximum exposure
            # depends on frame rate. Exposure will be automatically decreased when necessary.
            print("Set frame rate to 60 Hz")
            tracker.set_framerate(60)
            print("Frame rate set to: " + str(tracker.get_framerate()) + " Hz\n")
            print("Check exposure: " + str(tracker.get_exposure()) + " s")

            # Check new exposure range
            min, max = tracker.get_exposure_range()
            print("New exposure range: " + str(min) + " s - " + str(max) + " s")
            print("***************************\n")

            # Set exposure half-way
            exposure_half = min + (max - min) / 2.0
            print("Set exposure half way: " + str(exposure_half) + " s")
            tracker.set_exposure(exposure_half)
            print("Check new exposure: " + str(tracker.get_exposure()) + " s")
            print("***************************\n")

            while running:
                time.sleep(0.1)

    except psterrors.TrackerError as err:
        # Catch TrackerError and print error message
        print(err.message)

if __name__ == "__main__":
    main()
