﻿// Copyright PS-Tech B.V. All Rights Reserved.

using System;
using System.Threading;
using PSTech.Pstsdk;

namespace PSTech.Minimal
{
    /// <summary>Minimal example of the PST SDK</summary>
    /// <remarks>
    /// This is a bare minimum example showing how to connect to the PST SDK and how to
    /// register a listener to receive data.
    /// Note that for simplicity reasons, this example does not take the recommended safety
    /// precautions to shut down the PST Tracker when the application shuts down. In an actual
    /// implementation, please follow the safety precautions recommended in the "Safe Tracker Termination"
    /// section of the "Using PST SDK" page of the PST SDK manual. Examples of how to implement the safety
    /// precautions can be found in the other examples.
    /// </remarks>
    public class Minimal
    {
        /// Implementation of the PSTech.Pstsdk.Tracker.TrackerListener class to receive tracking data.
        class MyTrackerListener : TrackerListener
        {
            /// <summary>Implementation of a tracker data callback function</summary>
            /// <remarks>
            /// The OnTrackerData() callback function receives the data as soon as it becomes
            /// available.
            /// </remarks>
            /// <param name="data">Object containing tracking information retrieved from tracker</param>
            /// <param name="status">Status message reported by the tracker.</param>
            public override void OnTrackerData(TrackerData data, ErrorStatus status)
            {
                // Do something with the tracker data in data
            }
        }

        static void Main(string[] args)
        {

            // Implement error handling of PSTech.TrackerException exceptions to prevent
            // improper PST Tracker shutdown on errors.
            try
            {
                // Create an instance of the Tracker object using the default configuration path and file names.
                Tracker tracker = new Tracker();

                //Create tracker listener with callback functions for data and/or mode updates.
                TrackerListener listener = new MyTrackerListener();

                // Register the TrackerListener object to the tracker server.
                tracker.AddTrackerListener(ref listener);

                // Start the tracker server.
                tracker.StartTracker();

                // Wait for 10 seconds, allowing for the detection of tracking targets.
                Thread.Sleep(10000);
            }
            catch (Exception e)
            {
                // Do something with the error.
            }
            finally
            {
                // Make sure that the connection to the PST Tracker is shut down properly.
                Tracker.Shutdown();
            }
        }
    }
}
