/**
* images.c
* Copyright PS-Tech B.V. All Rights Reserved.
*
* This example shows how to enable image transfer on the PST Tracker and how to use
* the PST SDK to retrieve images. Images are 8 bit grayscale and are stored as an
* unsigned char* without memory alignment or padding.
*
*/

#ifdef WIN32
#include <windows.h>
#else
#include <unistd.h>
#endif

/*
* Define handler functions required to ensure a clean shutdown of the PST Tracker when the
* application is terminated.
*/

static void Exithandler(int sig);

#ifdef WIN32
BOOL WINAPI ConsoleHandler(DWORD CEvent)
{
    Exithandler(CEvent);
    return TRUE;
}
#endif

/* End of handler functions */

#include "pstsdk_c.h"
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <signal.h>

/* Control variable for main loop */
static sig_atomic_t running = 1;

/* Number of data points to grab before application termination */
static const uint32_t numberOfSamplesToGrab = 1000;

/*
* Implementation of a tracker callback function.
* The OnTrackerData() callback function receives the data as soon as it becomes
* available.
*/
void OnTrackerData(const PstTrackerData* tracker_data, EPstErrorStatus status)
{
    static uint32_t samplesGrabbed = 0;
    if (samplesGrabbed++ >= numberOfSamplesToGrab)
        running = 0;

    // Do something with the received data.
}

/*
* Implement the exit handler to shut-down the PST Tracker connection on application termination.
*/
static void Exithandler(int sig)
{
    pst_sdk_shutdown();
    exit(0);
}

// Print the last error message.
void PrintLastErrorMessage()
{
    char* last_error_message = NULL;
    EPstErrorStatus error_status = pst_alloc_and_get_last_error_message(&last_error_message);
    if (error_status != PST_ERROR_STATUS_OK)
    {
        last_error_message = "Failed to allocate memory error.";
    }
    printf("last error message: %s \n", last_error_message);
    pst_free(last_error_message);
}

// Check error status and shutdown tracker upon error.
void CheckErrorCode(EPstErrorStatus status)
{
    if (status != PST_ERROR_STATUS_OK)
    {
        PrintLastErrorMessage();
        pst_sdk_shutdown();
        exit(status);
    }
}

int main(int argc, char *argv[])
{
    // Register the exit handler with the application
#ifdef WIN32
    SetConsoleCtrlHandler((PHANDLER_ROUTINE)ConsoleHandler, TRUE);
#else
    signal(SIGTERM, Exithandler);
    signal(SIGKILL, Exithandler);
    signal(SIGQUIT, Exithandler);
    signal(SIGINT, Exithandler);
#endif

    PstTracker ctracker;

#ifdef WIN32
    // Create an instance of the Tracker object using the default configuration path and file names.
    CheckErrorCode(pst_tracker_init(&ctracker));
#else
    // On Linux, specify the type of grabber that needs to be used as the last parameter:
    // "basler_ace" for PST HD or "basler_dart" for PST Pico
    CheckErrorCode(pst_tracker_init4(&ctracker, "", "config.cfg", "models.db", argv[1]));
#endif

    char* version_string;
    CheckErrorCode(pst_tracker_alloc_and_get_version_info(&ctracker, &version_string));
    // Print version number of the tracker server being used.
    printf("Running PST Server version %s \n", version_string);
    pst_free(version_string);

    // Register the OnTrackerData callback function to the tracker server.
    CheckErrorCode(pst_tracker_add_tracker_data_callback(&ctracker, OnTrackerData));

    // Start the tracker server.
    CheckErrorCode(pst_tracker_start(&ctracker));

    // Perform a system check to see if the tracker server is running OK and print the result.
    printf("System check: %i \n\n", (int)pst_tracker_system_check(&ctracker));
    printf("***************************\n\n");

    // Set the frame rate to 60 Hz.
    CheckErrorCode(pst_tracker_set_framerate(&ctracker, 60.0));
    double fps;
    CheckErrorCode(pst_tracker_get_framerate(&ctracker, &fps));
    printf("Current frame rate: %f Hz\n\n", fps);

    // In order to start receiving images, enable image transfer. When image transfer is disabled,
    // the vector of images returned by GetImage will be empty.
    pst_tracker_enable_image_transfer(&ctracker);

    // The standard PST trackers will run at a reduced frame rate of 30 Hz when image transfer is enabled.
    // However, since this frame rate is temporary for as long as image transfer is enabled, that frame rate
    // will not be reported as the current frame rate.
    CheckErrorCode(pst_tracker_get_framerate(&ctracker, &fps));
    printf("Enabled image transfer. Current frame rate: %f Hz\n\n", fps);
    printf("***************************\n\n");

    // Try to capture 100 images.
    for (int i = 0; i < 100; ++i)
    {
        //Make sure to initialize the PstImage with ::pst_image_init before using pst_tracker_get_pst_image function.
        PstImage images;
        pst_image_init(&images);

        // Get a reference to the last grabbed image in the PstImage struct.
        // Note that enabling image transfer takes some time. While image transfer is being enabled,
        // the images vector in the PstImage struct will be empty.
        CheckErrorCode(pst_tracker_get_pst_image(&ctracker, &images));
        printf("Retrieval operation successful!\n");
        printf("Retrieved %zu image(s) of size: %zu X %zu \n\n", images.number_of_images, images.width, images.height);
        // Do something with the image.

        // Make sure to destroy PstImage after using it.
        pst_image_destroy(&images);
        // Don't request images too fast, wait for around 1/60 seconds
#ifdef WIN32
        Sleep(17);
#else
        usleep(17000);
#endif
    }
    // Wait for 5 seconds, since this is > 4 seconds, image transfer will be disabled automatically.
    printf("Waiting 5 seconds for image transfer to be automatically disabled...\n\n");
#ifdef WIN32
    Sleep(5000);
#else
    sleep(5);
#endif

    // Try to grab one image. Since image retrieval timed out, it should return an empty image vector.
    PstImage images;
    pst_image_init(&images);
    CheckErrorCode(pst_tracker_get_pst_image(&ctracker, &images));
    printf("Retrieval operation successful!\n");
    printf("Retrieved %zu image(s) of size: %zu X %zu \n\n", images.number_of_images, images.width, images.height);

    // Main loop, wait for auto-termination.
    while (running == 1)
    {
#ifdef WIN32
        Sleep(100);
#else
        usleep(100000);
#endif
    }

    // Make sure that the connection to the PST Tracker is shut down properly.
    pst_tracker_destroy(&ctracker);

    // Pause command line to see results.
    printf("Press enter to continue...\n");
    getchar();

    return 0;
}
