﻿// Copyright PS-Tech B.V. All Rights Reserved.

using System;
using System.Runtime.InteropServices;
using System.Threading;
using PSTech.Pstsdk;

namespace PsTech.TrackingTarget
{
    /// <summary>Tracking target example of the PST SDK</summary>
    /// <remarks>
    /// This example shows how to work with tracking targets using the PST SDK.
    /// Note that at this moment tracking targets can not be trained or
    /// imported using the PST SDK. In order to add new tracking targets, please
    /// use the PST Client together with the PSTech.Pstsdk.Tracker.EnableSharedMemory()
    /// function, or use the stand-alone PST Server to configure the tracking targets.
    /// </remarks>
    public class TrackingTarget
    {
        //Registering control handler function.
        [DllImport("Kernel32")]
        private static extern bool SetConsoleCtrlHandler(ConsoleEventHandler handler, bool add);

        private delegate bool ConsoleEventHandler(CtrlType sig);
        static ConsoleEventHandler ConsoleCtrlEventHandler;

        public enum CtrlType
        {
            CtrlCEvent = 0,
            CtrlBreakEvent = 1,
            CtrlCloseEvent = 2,
            CtrlLogOffEvent = 5,
            CtrlShutdownEvent = 6,
        }

        /// <summary>Define handler functions required to ensure a clean shutdown of the PST Tracker when the application is terminated.</summary>
        /// <param name="sig">Signal type</param>
        /// <returns>Signal handling status</returns>
        private static bool ConsoleHandler(CtrlType sig)
        {
            Exithandler((int)sig);
            return true;
        }

        // Control variable for main loop.
        static bool Running = true;

        // Number of data points to grab before application termination.
        static int NumberOfSamplesToGrab = 100;
        static int SamplesGrabbed = 0;

        /// Implementation of the PSTech.Pstsdk.Tracker.TrackerListener class to receive tracking data.
        class MyTrackerListener : TrackerListener
        {
            /// <summary>Implementation of a tracker data callback function</summary>
            /// <remarks>
            /// The OnTrackerData() callback function receives the data as soon as it becomes
            /// available.
            /// </remarks>
            /// <param name="data">Object containing tracking information retrieved from tracker</param>
            /// <param name="status">Status message reported by the tracker.</param>
            public override void OnTrackerData(TrackerData data, ErrorStatus status)
            {
                if (SamplesGrabbed++ >= NumberOfSamplesToGrab)
                {
                    Running = false;
                }
                foreach (TargetPose pose in data.TargetPoseList)
                {
                    Console.Write("Detected " + pose.Target.Name + "\n");
                }

                // Do something with the received data.
            }
        }

        /// <summary>
        /// Implement the exit handler to shut-down the PST Tracker connection on application termination.
        /// </summary>
        /// <param name="sig">Signal type.</param>
        static void Exithandler(int sig)
        {
            Tracker.Shutdown();
            Running = false;
        }

        static int Main(string[] args)
        {
            // Register the exit handler with the application
            ConsoleCtrlEventHandler += new ConsoleEventHandler(ConsoleHandler);
            SetConsoleCtrlHandler(ConsoleCtrlEventHandler, true);

            // Implement error handling of PSTech.TrackerException exceptions to prevent
            // improper PST Tracker shutdown on errors.
            try
            {
                // Create an instance of the Tracker object using the default configuration path and file names.
                Tracker tracker = new Tracker();

                // Print version number of the tracker server being used.
                Console.Write("Running PST Server version " + tracker.GetVersionInfo() + "\n");

                //Create tracker listener with callback functions for data and/or mode updates.
                TrackerListener listener = new MyTrackerListener();

                // Register the TrackerListener object to the tracker server.
                tracker.AddTrackerListener(ref listener);

                Console.Write("Put the Reference card in front of the PST in order to see tracking results.\n\n");

                // Start the tracker server.
                tracker.StartTracker();

                // Perform a system check to see if the tracker server is running OK and print the result.
                Console.Write("System check: " + tracker.Systemcheck() + "\n\n");

                // Retrieve the list of registered tracking targets and print their names and current status (active or not).
                TargetStatus[] targets = tracker.GetTargetList();
                Console.Write("Found " + targets.Length + " tracking targets:\n");
                foreach (TargetStatus t in targets)
                {
                    Console.Write(t.Target.Name + "\t" + t.Status + "\n");
                }
                Console.Write("\n");

                // Enable the Reference target. Note that this will cause a PSTech.TrackerException in case the Reference
                // target has not been created. The Reference target can be created using the PST Client.
                tracker.SetTargetStatus("Reference", true);

                // Get the 3D marker positions making up the Reference device and display them.
                // Note that this will cause a PSTech.TrackerException in case the Reference target has not been created.
                TargetMarker[] markers = tracker.GetTargetMarkers("Reference");
                Console.Write("3D marker positions making up the Reference target:\n");
                foreach (TargetMarker marker in markers)
                {
                    Console.Write("x: " + marker.Coordinates[0] + "\t" + "y: " + marker.Coordinates[1] + "\t" + "z: " + marker.Coordinates[2] + "\n");
                }
                Console.Write("\n");

                // Main loop, wait for auto-termination.
                while (Running)
                {
                    Thread.Sleep(100);
                }

            }
            catch (Exception e)
            {
                // Catch PSTech.TrackerException exceptions and print error messages.
                Console.Write(e + "\n");
            }
            finally
            {
                // Make sure that the connection to the PST Tracker is shut down properly.
                Tracker.Shutdown();
            }

            // Pause command line to see results.
            Console.Write("Press enter to continue...\n");
            Console.Read();

            return 0;
        }
    }
}
