/**
 * sharedmemory.c
 * Copyright PS-Tech B.V. All Rights Reserved.
 *
 * This example shows how to activate the shared memory communication pipeline
 * that enables communication of the PST Client application through the PST SDK.
 * Note that for the shared memory pipeline to function, the application has to
 * run with elevated access (administrator rights). After enabling shared memory,
 * the PST Client application can be used to download calibration information and
 * manage tracking targets.
 *
 */

#ifdef WIN32
#include <windows.h>
#else
#include <unistd.h>
#endif

/*
 * Define handler functions required to ensure a clean shutdown of the PST Tracker
 * when the application is terminated.
 */

static void Exithandler(int sig);

#ifdef WIN32
BOOL WINAPI ConsoleHandler(DWORD CEvent)
{
    Exithandler(CEvent);
    return TRUE;
}
#endif

/* End of handler functions */

#include "pstsdk_c.h"
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

/*
* Implement the exit handler to shut-down the PST Tracker connection on application termination.
*/
static void Exithandler(int sig)
{
    pst_sdk_disable_shared_memory();
    pst_sdk_shutdown();
    exit(0);
}

// Print the last error message.
void PrintLastErrorMessage()
{
    char* last_error_message = NULL;
    EPstErrorStatus error_status = pst_alloc_and_get_last_error_message(&last_error_message);
    if (error_status != PST_ERROR_STATUS_OK)
    {
        last_error_message = "Failed to allocate memory error.";
    }
    printf("last error message: %s \n", last_error_message);
    pst_free(last_error_message);
}

// Check error status and shutdown tracker upon error.
void CheckErrorCode(EPstErrorStatus status)
{
    if (status != PST_ERROR_STATUS_OK)
    {
        PrintLastErrorMessage();
        pst_sdk_shutdown();
        exit(status);
    }
}

int main(int argc, char *argv[])
{
    // Register the exit handler with the application
#ifdef WIN32
    SetConsoleCtrlHandler((PHANDLER_ROUTINE)ConsoleHandler, TRUE);
#else
    signal(SIGTERM, Exithandler);
    signal(SIGKILL, Exithandler);
    signal(SIGQUIT, Exithandler);
    signal(SIGINT, Exithandler);
#endif

    PstTracker ctracker;
    
#ifdef WIN32
    // Create an instance of the Tracker object using the default configuration path and file names.
    CheckErrorCode(pst_tracker_init(&ctracker));
#else
    // On Linux, specify the type of grabber that needs to be used as the last parameter:
    // "basler_ace" for PST HD or "basler_dart" for PST Pico
    CheckErrorCode(pst_tracker_init4(&ctracker, "", "config.cfg", "models.db", argv[1]));
#endif

    // Start the tracker server.
    CheckErrorCode(pst_tracker_start(&ctracker));

    // Enable the shared memory pipeline to enable client connections.
    CheckErrorCode(pst_sdk_enable_shared_memory());

    printf("Shared memory server initialized. Start the PST Client application to create a connection.\n");

    // Wait for one minute before terminating this application.
#ifdef WIN32
    Sleep(60000);
#else
    sleep(60);
#endif

    // Disable the shared memory pipeline. If the PST Client is still connected at this point,
    // it will try to reconnect and otherwise exit.
    pst_sdk_disable_shared_memory();

    // Make sure that the connection to the PST Tracker is shut down properly.
    pst_tracker_destroy(&ctracker);

    return 0;
}
