/**
 * exposure.c
 * Copyright PS-Tech B.V. All Rights Reserved.
 *
 * This example shows how to adjust exposure settings using the PST SDK. It shows how to
 * change exposure settings based on frame rate and the available exposure range for a
 * certain PST Tracker at a certain frame rate.
 *
 */

#ifdef WIN32
#include <windows.h>
#else
#include <unistd.h>
#endif

/*
 * Define handler functions required to ensure a clean shutdown of the PST Tracker when the
 * application is terminated.
 */

static void Exithandler(int sig);

#ifdef WIN32
BOOL WINAPI ConsoleHandler(DWORD CEvent)
{
    Exithandler(CEvent);
    return TRUE;
}
#endif

/* End of handler functions */

#include "pstsdk_c.h"
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <signal.h>

/* Control variable for main loop */
static sig_atomic_t running = 1;

/* Number of data points to grab before application termination */
static const uint32_t numberOfSamplesToGrab = 1000;

/*
 * Implementation of a tracker callback function.
 * The OnTrackerData() callback function receives the data as soon as it becomes
 * available and prints the tracking target pose to the command line.
 */
void OnTrackerData(const PstTrackerData* tracker_data, EPstErrorStatus status)
{
    static uint32_t samplesGrabbed = 0;
    if (samplesGrabbed++ >= numberOfSamplesToGrab)
        running = 0;

    // Do something with the received data.
}

/*
* Implement the exit handler to shut-down the PST Tracker connection on application termination.
*/
static void Exithandler(int sig)
{
    pst_sdk_shutdown();
    running = 0;
}

// Print the last error message.
void PrintLastErrorMessage()
{
    char* last_error_message = NULL;
    EPstErrorStatus error_status = pst_alloc_and_get_last_error_message(&last_error_message);
    if (error_status != PST_ERROR_STATUS_OK)
    {
        last_error_message = "Failed to allocate memory error.";
    }
    printf("last error message: %s \n", last_error_message);
    pst_free(last_error_message);
}

// Check error status and shutdown tracker upon error.
void CheckErrorCode(EPstErrorStatus status)
{
    if (status != PST_ERROR_STATUS_OK)
    {
        PrintLastErrorMessage();
        pst_sdk_shutdown();
        exit(status);
    }
}

int main(int argc, char *argv[])
{

    // Register the exit handler with the application
#ifdef WIN32
    SetConsoleCtrlHandler((PHANDLER_ROUTINE)ConsoleHandler, TRUE);
#else
    signal(SIGTERM, Exithandler);
    signal(SIGKILL, Exithandler);
    signal(SIGQUIT, Exithandler);
    signal(SIGINT, Exithandler);
#endif

    PstTracker ctracker;

#ifdef WIN32
    // Create an instance of the Tracker object using the default configuration path and file names.
    CheckErrorCode(pst_tracker_init(&ctracker));
#else
    // On Linux, specify the type of grabber that needs to be used as the last parameter:
    // "basler_ace" for PST HD or "basler_dart" for PST Pico
    CheckErrorCode(pst_tracker_init4(&ctracker, "", "config.cfg", "models.db", argv[1]));
#endif

    char* version_string;
    CheckErrorCode(pst_tracker_alloc_and_get_version_info(&ctracker, &version_string));
    // Print version number of the tracker server being used.
    printf("Running PST Server version %s \n", version_string);
    pst_free(version_string);

    // Register the OnTrackerData callback function to the tracker server.
    CheckErrorCode(pst_tracker_add_tracker_data_callback(&ctracker, OnTrackerData));

    // Start the tracker server.
    CheckErrorCode(pst_tracker_start(&ctracker));

    // Perform a system check to see if the tracker server is running OK and print the result.
    printf("System check: %i \n\n", (int)pst_tracker_system_check(&ctracker));
    printf("***************************\n\n");

    // Set the frame rate to 30 Hz.
    CheckErrorCode(pst_tracker_set_framerate(&ctracker, 30.0));

    // Print the new frame rate to see if it was set correctly. Note that for PST HD and Pico
    // trackers the frame rate actually being set can differ from the value provided to pst_tracker_set_framerate().
    double fps;
    CheckErrorCode(pst_tracker_get_framerate(&ctracker, &fps));
    printf("Frame rate set to %f Hz\n", fps);

    // Query available exposure range for the current frame rate and try setting the maximum exposure value
    double min, max;
    CheckErrorCode(pst_tracker_get_exposure_range(&ctracker, &min, &max));
    printf("Exposure range: %f s - %f s\n", min, max);
    printf("Set exposure to %f \n", max);
    CheckErrorCode(pst_tracker_set_exposure(&ctracker, max));
    double exposure;
    CheckErrorCode(pst_tracker_get_exposure(&ctracker, &exposure));
    printf("Check new exposure: %f s\n\n", exposure);
    printf("***************************\n\n");

    // Increase frame rate and check exposure value. For PST HD and PST Pico trackers, maximum exposure
    // depends on frame rate. Exposure will be automatically decreased when necessary.
    printf("Set frame rate to 60 Hz\n");
    CheckErrorCode(pst_tracker_set_framerate(&ctracker, 60.0));
    CheckErrorCode(pst_tracker_get_framerate(&ctracker, &fps));
    printf("Frame rate set to %f Hz\n", fps);
    CheckErrorCode(pst_tracker_get_exposure(&ctracker, &exposure));
    printf("Check exposure: %f s\n", exposure);

    // Check new exposure range
    CheckErrorCode(pst_tracker_get_exposure_range(&ctracker, &min, &max));
    printf("New exposure range %f s - %f s\n\n", min, max);
    printf("***************************\n\n");

    // Set exposure half-way
    double exposureHalf = min + (max - min) / 2.0;
    printf("Set exposure half way: %f s\n", exposureHalf);
    CheckErrorCode(pst_tracker_set_exposure(&ctracker, exposureHalf));
    CheckErrorCode(pst_tracker_get_exposure(&ctracker, &exposure));
    printf("New exposure: %f s\n\n", exposure);
    printf("***************************\n\n");

    // Main loop, wait for auto-termination.
    while (running == 1)
    {
#ifdef WIN32
        Sleep(100);
#else
        usleep(100000);
#endif
    }

    // Make sure that the connection to the PST Tracker is shut down properly.
    pst_tracker_destroy(&ctracker);

    // Pause command line to see results.
    printf("Press enter to continue...\n");
    getchar();

    return 0;
}
