/**
 * trackingtarget.c
 * Copyright PS-Tech B.V. All Rights Reserved.
 *
 * This example shows how to work with tracking targets using the PST SDK.
 * Note that at this moment tracking targets can not be trained or
 * imported using the PST SDK. In order to add new tracking targets, please
 * use the PST Client together with the pst_tracker_enable_shared_memory
 * function, or use the stand-alone PST Server to configure the tracking targets.
 *
 */

#ifdef WIN32
#include <windows.h>
#else
#include <unistd.h>
#endif

/*
 * Define handler functions required to ensure a clean shutdown of the PST Tracker when the
 * application is terminated.
 */

static void Exithandler(int sig);

#ifdef WIN32
BOOL WINAPI ConsoleHandler(DWORD CEvent)
{
    Exithandler(CEvent);
    return TRUE;
}
#endif

/* End of handler functions */

#include "pstsdk_c.h"
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <signal.h>

/* Control variable for main loop */
static sig_atomic_t running = 1;

/* Number of data points to grab before application termination */
static const uint32_t numberOfSamplesToGrab = 100;

/*
 * Implementation of a tracker callback function.
 * The OnTrackerData() callback function receives the data as soon as it becomes
 * available and prints the tracking target pose to the command line.
 */
void OnTrackerData(const PstTrackerData* tracker_data, EPstErrorStatus status)
{
    static uint32_t samplesGrabbed = 0;
    if (samplesGrabbed++ >= numberOfSamplesToGrab)
    {
        running = 0;
    }

    for (int d = 0; d < tracker_data->number_of_targets; ++d)
    {
        printf("Detected %s \n", tracker_data->targetlist[d].target.name);
    }
}

/*
* Implement the exit handler to shut-down the PST Tracker connection on application termination.
*/
static void Exithandler(int sig)
{
    pst_sdk_shutdown();
    running = 0;
}

// Print the last error message.
void PrintLastErrorMessage()
{
    char* last_error_message = NULL;
    EPstErrorStatus error_status = pst_alloc_and_get_last_error_message(&last_error_message);
    if (error_status != PST_ERROR_STATUS_OK)
    {
        last_error_message = "Failed to allocate memory error.";
    }
    printf("last error message: %s \n", last_error_message);
    pst_free(last_error_message);
}

// Check error status and shutdown tracker upon error.
void CheckErrorCode(EPstErrorStatus status)
{
    if (status != PST_ERROR_STATUS_OK)
    {
        PrintLastErrorMessage();
        pst_sdk_shutdown();
        exit(status);
    }
}

int main(int argc, char *argv[])
{
    // Register the exit handler with the application
#ifdef WIN32
    SetConsoleCtrlHandler((PHANDLER_ROUTINE)ConsoleHandler, TRUE);
#else
    signal(SIGTERM, Exithandler);
    signal(SIGKILL, Exithandler);
    signal(SIGQUIT, Exithandler);
    signal(SIGINT, Exithandler);
#endif

    PstTracker ctracker;
    
#ifdef WIN32
    // Create an instance of the Tracker object using the default configuration path and file names.
    CheckErrorCode(pst_tracker_init(&ctracker));
#else
    // On Linux, specify the type of grabber that needs to be used as the last parameter:
    // "basler_ace" for PST HD or "basler_dart" for PST Pico
    CheckErrorCode(pst_tracker_init4(&ctracker, "", "config.cfg", "models.db", argv[1]));
#endif

    char* version_string;
    CheckErrorCode(pst_tracker_alloc_and_get_version_info(&ctracker, &version_string));
    // Print version number of the tracker server being used.
    printf("Running PST Server version %s \n", version_string);
    pst_free(version_string);

    // Register the OnTrackerData callback function to the tracker server.
    CheckErrorCode(pst_tracker_add_tracker_data_callback(&ctracker, &OnTrackerData));

    printf("Put the Reference card in front of the PST in order to see tracking results.\n\n");

    // Start the tracker server.
    CheckErrorCode(pst_tracker_start(&ctracker));

    // Perform a system check to see if the tracker server is running OK and print the result.
    printf("System check: %i \n", pst_tracker_system_check(&ctracker));

    // Retrieve the list of registered tracking targets and print their names and current status (active or not).
    PstTargetStatus* statuses;
    size_t number_of_statuses;
    CheckErrorCode(pst_tracker_alloc_and_get_target_list(&ctracker, &statuses, &number_of_statuses));
    printf("Found %zu tracking targets:\n", number_of_statuses);
    for (int i = 0; i < number_of_statuses; ++i)
    {
        printf("%s\t%s\n", statuses[i].target.name, statuses[i].status ? "true" : "false");
    }
    printf("\n");
    pst_free(statuses);

    // Enable the Reference target. Note that this will return an error in case the Reference
    // target has not been created. The Reference target can be created using the PST Client.
    CheckErrorCode(pst_tracker_set_target_status(&ctracker, "Reference", true));


    PstTargetMarkers marker_list;
    //Make sure to initialize the PstImage with ::PstTargetMarkers before using pst_tracker_get_target_markers function.
    pst_target_markers_init(&marker_list);

    // Get the 3D marker positions making up the Reference device and display them.
    // Note that this will cause an error in case the Reference target has not been created.
    CheckErrorCode(pst_tracker_get_target_markers(&ctracker, "Reference", &marker_list));
    printf("3D marker positions making up the Reference target:\n");
    for (int i = 0; i < marker_list.number_of_markers; ++i)
    {
        printf("x: %f\ty: %f\tz: %f\n", marker_list.markers[i].coordinates[0], marker_list.markers[i].coordinates[1], marker_list.markers[i].coordinates[2]);
    }
    printf("\n");

    //Make sure to destroy PstTargetMarkers after using it.
    pst_target_markers_destroy(&marker_list);

    // Main loop, wait for auto-termination.
    while (running == 1)
    {
#ifdef WIN32
        Sleep(100);
#else
        usleep(100000);
#endif
    }

    // Make sure that the connection to the PST Tracker is shut down properly.
    pst_tracker_destroy(&ctracker);

    // Pause command line to see results.
    printf("Press enter to continue...\n");
    getchar();

    return 0;
}
