/**
 * minimal.c
 * Copyright PS-Tech B.V. All Rights Reserved.
 *
 * This is a bare minimum example showing how to connect to the PST SDK and how to
 * register a listener to receive data.
 *
 * Note that for simplicity reasons, this
 * example does not take the recommended safety precautions to shut down the PST
 * Tracker when the application shuts down. In an actual implementation, please
 * follow the safety precautions recommended in the "Safe Tracker Termination"
 * section of the "Using PST SDK" page of the PST SDK manual. Examples of how to implement
 * the safety precautions can be found in the other examples.
 *
 */


#include "pstsdk_c.h"
#include <stdio.h>
#include <stdlib.h>
#ifdef WIN32
#include <Windows.h>
#else
#include <unistd.h>
#endif

 /*
  * Implementation of a tracker callback function.
  * The OnTrackerData() callback function receives the data as soon as it becomes
  * available and prints the tracking target pose to the command line.
  */
void OnTrackerData(const PstTrackerData* tracker_data, EPstErrorStatus status)
{
    // Do something with the tracker data in tracker_data
}

// Print the last error message.
void PrintLastErrorMessage()
{
    char* last_error_message = NULL;
    EPstErrorStatus error_status = pst_alloc_and_get_last_error_message(&last_error_message);
    if (error_status != PST_ERROR_STATUS_OK)
    {
        last_error_message = "Failed to allocate memory error.";
    }
    printf("last error message: %s \n", last_error_message);
    pst_free(last_error_message);
}

// Check error status and shutdown tracker upon error.
void CheckErrorCode(EPstErrorStatus status)
{
    if (status != PST_ERROR_STATUS_OK)
    {
        PrintLastErrorMessage();
        pst_sdk_shutdown();
        exit(status);
    }
}

int main(int argc, char* argv[])
{
    PstTracker ctracker;

#ifdef WIN32
    // Create an instance of the Tracker object using the default configuration path and file names.
    CheckErrorCode(pst_tracker_init(&ctracker));
#else
    // On Linux, specify the type of grabber that needs to be used as the last parameter:
    // "basler_ace" for PST HD or "basler_dart" for PST Pico
    CheckErrorCode(pst_tracker_init4(&ctracker, "", "config.cfg", "models.db", argv[1]));
#endif

    // Register the OnTrackerData callback function to the tracker server.
    CheckErrorCode(pst_tracker_add_tracker_data_callback(&ctracker, OnTrackerData));

    // Start the tracker server.
    CheckErrorCode(pst_tracker_start(&ctracker));

    // Wait for 10 seconds, allowing for the detection of tracking targets.
#ifdef WIN32
    Sleep(10000);
#else
    sleep(10);
#endif
    // Make sure that the connection to the PST Tracker is shut down properly.
    pst_tracker_destroy(&ctracker);

    return 0;
}

