// Copyright PS-Tech B.V. All Rights Reserved.
#pragma once

#if (defined(_WIN32) || defined(_MSC_VER) || defined(__MWERKS__) || defined(__BORLANDC__))
#    ifdef pstsdk_c_EXPORTS
#      define PST_C_EXPORT	__declspec(dllexport)
#    else
#      define PST_C_EXPORT	__declspec(dllimport)
#    endif
#else
#    define PST_C_EXPORT
#endif

#ifdef __cplusplus
extern "C" {
#else
#include <stdbool.h>
#include <stddef.h>
#endif

/**
* @defgroup pstsdkc PST SDK C API
* This library defines the C API for the PST. Applications implementing this API
* get direct access to the PST to receive tracking data and set tracker parameters.
* Using this library, it is also possible to initialize a shared memory communication
* pipeline to connect to the PST Client application and initialize a REST server for
* HTTP-based communication.
*
* Some functions allocate memory to store objects. These functions will contain
* _alloc_ or _init_ in their names. As is conventional an _alloc_ should always be
* paired with a free. If a corresponding free function is defined in this API, this
* function should be used to release memory, otherwise a normal free should be
* sufficient. Likewise every _init_ should be paired with a _destroy_.
*
* @{
*/

/**
 * @brief Tracker error messages enum class.
 *
 * This enum class lists all error states that can be reported by the tracker.
 */
typedef enum EPstErrorStatus
{
    PST_ERROR_STATUS_OK                                 = 0,  /**< No error. */
    PST_ERROR_STATUS_TRACKER_ERROR                      = 1,  /**< General tracker error. */
    PST_ERROR_STATUS_NOT_INITIALIZED_ERROR              = 2,  /**< A call was made to an uninitialized tracker. */
    PST_ERROR_STATUS_NOT_SUPPORTED_ERROR                = 3,  /**< The requested functionality is not supported by this tracker. */
    PST_ERROR_STATUS_OUT_OF_RANGE_ERROR                 = 4,  /**< The supplied value is outside of the accepted range. */
    PST_ERROR_STATUS_NOT_FOUND_ERROR                    = 5,  /**< The requested data could not be found. */
    PST_ERROR_STATUS_HTTP_ERROR                         = 6,  /**< An HTTP error was thrown by the REST server. */
    PST_ERROR_STATUS_FAILED_TO_ALLOCATE_MEMORY_ERROR    = 7,  /**< Failed to allocate memory. */
    PST_ERROR_STATUS_NULLPTR_ERROR                      = 8,  /**< The supplied tracker was a null pointer. */
    PST_ERROR_STATUS_IMAGE_RETRIEVAL_FAILED_ERROR       = 9,  /**< Image retrieval failed. */
    PST_ERROR_STATUS_INVALID_DATA_ERROR                 = 10, /**< Data provided did not match the requirements. */
    PST_ERROR_STATUS_ALREADY_EXISTS_ERROR               = 11, /**< The provided object already exists. */
    PST_ERROR_STATUS_JSON_PARSING_ERROR                 = 12, /**< An error occurred while processing JSON information. */
    
} EPstErrorStatus;

/**
 * @brief Tracker status messages enum class.
 *
 * This enum class lists all status messages that can be reported by the tracker.
 *
 * @see pst_tracker_system_check()
 */
typedef enum EPstStatusMessage
{
    PST_STATUSMESSAGE_OK                                = 0,   /**< The system is running OK. */
    PST_STATUSMESSAGE_NOT_INITIALIZED                   = 1,   /**< Tracker failed to initialize correctly. */
    PST_STATUSMESSAGE_DISCONNECTED                      = 2,   /**< The grabber has been disconnected. */
    PST_STATUSMESSAGE_ERR_GENERAL                       = 3,   /**< Unspecified grabber error. */
    PST_STATUSMESSAGE_ERR_TIMEOUT                       = 4,   /**< Grabber timeout error. */
    PST_STATUSMESSAGE_ERR_NOCAMS_FOUND                  = 5,   /**< Grabber could not detect any cameras. */
    PST_STATUSMESSAGE_ERR_NOTENOUGHTCAMS_FOUND          = 6,   /**< Grabber could not detect sufficient cameras. */
    PST_STATUSMESSAGE_ERR_INITERROR                     = 7,   /**< Grabber did not initialize correctly. */
    PST_STATUSMESSAGE_ERR_CANNOT_START_CAMS             = 8,   /**< Grabber could not start cameras. */
    PST_STATUSMESSAGE_ERR_CANNOT_SETUP_CAMS             = 9,   /**< Grabber failed setting up cameras. */
    PST_STATUSMESSAGE_ERR_NULL_TRACKER_HANDLE           = 10,  /**< Tracker not created or freed. */
    PST_STATUSMESSAGE_ERR_UNKNOWN_STATUS                = 11,  /**< Unknown status message. */
} EPstStatusMessage;

/**
 * @brief Tracker mode enum class.
 *
 * This enum class lists all the possible modes that the tracker can be in.
 *
 * @see pst_tracker_add_tracker_mode_callback()
 * @see pst_tracker_add_tracker_callbacks()
 * @see pst_tracker_remove_tracker_mode_callback()
 * @see pst_tracker_remove_tracker_callbacks()
 */
typedef enum EPstTrackerMode
{
    PST_MODE_LOWPOWER                     = 0,   /**< Tracker is in lowpower mode waiting for activity to resume */
    PST_MODE_SETUP                        = 1,   /**< Tracker is in setup mode */
    PST_MODE_TRACKING                     = 2,   /**< Tracker is in tracking mode */
    PST_MODE_TRAINING                     = 3,   /**< Tracker is in training mode */
    PST_MODE_DISCONNECT                   = 4,   /**< Tracker has been disconnected */
    PST_MODE_RECONNECT                    = 5,   /**< Tracker has been reconnected */
    PST_MODE_UNKNOWN                      = 6,   /**< Tracker is in an unknown state */
} EPstTrackerMode;

/**
 * @brief Images retrieved from the tracker.
 *
 * This struct contains pointers to grayscale images retrieved from the cameras inside the PST Tracker.
 *
 * @see pst_image_init()
 * @see pst_tracker_get_pst_image()
 * @see pst_image_destroy()
 * @see pst_tracker_enable_image_transfer()
 * @see pst_tracker_disable_image_transfer()
 */
typedef struct PstImage
{
    size_t width;               /**< Image width in pixels */
    size_t height;              /**< Image height in pixels */
    size_t number_of_images;    /**< Number of images that were retrieved */
    unsigned char** images;     /**< Array containing pointers to grayscale images for each camera */
} PstImage;

/**
 * @brief Tracking target information.
 *
 * This struct contains basic information about tracking targets known to the PST Tracker.
 * In order to be accessible, these targets have to be trained or imported using the PST-Client software.
 * The targets are stored in the model database.
 */
typedef struct PstTarget
{
    char name[128]; /**< User given name of the tracking target as listed in the PST-Client software */
    char uuid[37];  /**< Unique identifier, automatically assigned to the tracking target */
    int id;         /**< User provided tracking id assigned to the tacking target as listed in the PST-Client software */
} PstTarget;

/**
 * @brief Tracking target status.
 *
 * This struct provides status information about a tracking target.
 * The status is either true (active), in which case the object will be tracked when seen by the PST Tracker,
 * or false (inactive), in which case it will not be tracked by the PST Tracker.
 *
 * @see PstTarget
 * @see pst_tracker_alloc_and_get_target_list()
 * @see pst_tracker_set_target_status()
 * @see pst_tracker_get_target_status()
 */
typedef struct PstTargetStatus
{
    PstTarget target;   /**< Tracking target */
    bool status;        /**< Status of the tracking target (active or inactive) as listed in the PST-Client software */
} PstTargetStatus;

/**
 * @brief Estimated 3D marker location.
 *
 * This struct contains information about individual 3D markers that have been detected by the PST Tracker
 * but have not been estimated to be part of an active tracking target. As a single 3D point has no features
 * to distinguish it from another, points are given an identifier based on their previous motion. Note that
 * there is no guarantee that the identifier is consistent between sensor updates.
 *
 * @see PstTrackerData
 */
typedef struct PstPoint
{
    int id;             /**< Marker identifier */
    float position[3];  /**< 3D marker location */
} PstPoint;

/**
 * @brief Estimated tracking target pose.
 *
 * This struct contains the estimated pose information and matched points of a tracking target.
 * The pose is provided as a row-major 4x4 transformation matrix describing the pose of the tracking target in the coordinate system
 * defined in the PST-Client application (see the 'Reference coordinate system' Section in the PST Manual).
 * The matched points are provided as a list of 3D data points that have been detected by the tracker and are estimated be part of the Target.
 *
 * @see PstTarget
 * @see PstTrackerData
 */
typedef struct PstTargetPose
{
    PstTarget target;                   /**< Tracking target */
    float pose_matrix[16];              /**< Tracking target pose as a row-major 4x4 transformation matrix */
    size_t number_of_matched_points;    /**< The number of matched points */
    PstPoint* matched_points;           /**< Array of detected 3D markers matched to this tracking target */
} PstTargetPose;

/**
 * @brief 3D marker positions of a single marker of a tracking Target.
 *
 * 3D marker positions of a single marker of a tracking Target.
 *
 * @see PstTarget
 * @see PstTargetMarkers
 */
typedef struct PstTargetMarker
{
    float coordinates[3];   /**< 3D marker position of a tracking target */
} PstTargetMarker;

/**
 * @brief Collection struct for PstTargetMarker.
 *
 * Collection struct for PstTargetMarker.
 *
 * @see PstTarget
 * @see PstTargetMarker
 */
typedef struct PstTargetMarkers
{
    PstTargetMarker* markers;   /**< Array of tracking target 3D marker positions */
    size_t number_of_markers;   /**< Number of elements in the array markers */
} PstTargetMarkers;

/**
 * @brief Tracking information retrieved from tracker.
 *
 * This struct contains all tracking information recorded at a certain time instance. It provides a list of all
 * active tracking targets that have been detected by the PST Tracker, as well as a list of all 3D data points
 * that have been detected by the tracker and are estimated not to be part of any active tracking target.
 *
 * @see PstTargetPose
 * @see PstPoint
 * @see pst_tracker_add_tracker_data_callback()
 * @see pst_tracker_add_tracker_callbacks()
 * @see pst_tracker_get_single_measurement()
 */
typedef struct PstTrackerData
{
    double timestamp;           /**< Timestamp of the moment the cameras captured the data. The timestamp uses the system clock provided in seconds since system boot (Windows) or Epoch (Linux). */
    int seqnumber;              /**< Incremental data recording id */
    size_t number_of_targets;   /**< The number of targets in targetlist */
    size_t number_of_points;    /**< The number of points in datapoints */
    PstTargetPose* targetlist;  /**< Array of detected tracking targets */
    PstPoint* datapoints;       /**< Array of detected 3D markers not matched to a tracking target */
} PstTrackerData;

/**
 * @brief Main PST SDK struct for tracker communication.
 *
 * This struct allows communication with  the main PST SDK functionality. Review the Examples section to
 * learn how this struct should be used.
 */
typedef struct PstTracker
{
    void* tracker;
} PstTracker;

/**
 * @brief Struct to request urls for calibration information.
 *
 * This struct contains a list of urls of calibration files.
 * 
 * @see pst_tracker_get_connected_camera_urls
 * @see pst_tracker_get_uncalibrated_camera_urls
 */
typedef struct PstCameraURLs
{
    char** url_data;        /**< Array of urls for calibration files */
    size_t number_of_urls;  /**< Number of urls in url data */
} PstCameraURLs;

/**
 * @brief This function returns the last error message that was recorded by the system.
 *
 * This function returns the last error message that was recorded by the system. It will try to allocate memory
 * for and copy the error message in a pointer returned in cstring. It is important that cstring does not point
 * to allocated memory, otherwise a memory leak is introduced. Since memory is allocated for *cstring, ensure that
 * it is freed using pst_free() when it is no longer needed.
 *
 * @param[out] cstring - Pointer to a pointer to a memory location where the error message is allocated.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_FAILED_TO_ALLOCATE_MEMORY_ERROR.
 *
 * @see pst_free()
 */
PST_C_EXPORT EPstErrorStatus pst_alloc_and_get_last_error_message(char** cstring);

/**
 * @brief This function initializes the tracker object.
 *
 * This function initializes the tracker object. Camera calibration will be checked. When calibration files for the connected tracker
 * can not be found, a message with download links will be printed to the command line. Note that an initialized tracker object should
 * be destroyed with pst_tracker_destroy() before the tracker object is freed.
 *
 * @note Ensure that a tracker object is initialized only once.
 *
 * @param[in] ctracker The tracker object to be initialized.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NOT_INITIALIZED_ERROR, ::PST_ERROR_STATUS_TRACKER_ERROR.
 *
 * @see pst_tracker_destroy()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_init( PstTracker* ctracker);

/**
 * @brief This function initializes the tracker object.
 *
 * This function initializes the tracker object. Camera calibration will be checked. When calibration files for the connected tracker
 * can not be found, a message with download links will be printed to the command line. Note that an initialized tracker object should
 * be destroyed with pst_tracker_destroy() before the tracker object is freed.
 *
 * @note Ensure that a tracker object is initialized only once.
 *
 * @param[in] ctracker The tracker object to be initialized.
 * @param[in] path Path to the configuration directory to be used for storing camera calibrations, target database and configuration files. Use "" for default. (default: "%PROGRAMDATA%\PS-Tech\PST Iris" on Windows or ~/.pstiris on Linux)
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NOT_INITIALIZED_ERROR, ::PST_ERROR_STATUS_TRACKER_ERROR.
 *
 * @see pst_tracker_destroy()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_init1(PstTracker* ctracker, const char path[]);

/**
 * @brief This function initializes the tracker object.
 *
 * This function initializes the tracker object. Camera calibration will be checked. When calibration files for the connected tracker
 * can not be found, a message with download links will be printed to the command line. Note that an initialized tracker object should
 * be destroyed with pst_tracker_destroy() before the tracker object is freed.
 *
 * @note Ensure that a tracker object is initialized only once.
 *
 * @param[in] ctracker The tracker object to be initialized.
 * @param[in] path Path to the configuration directory to be used for storing camera calibrations, target database and configuration files. Use "" for default. (default: "%PROGRAMDATA%\PS-Tech\PST Iris" on Windows or ~/.pstiris on Linux)
 * @param[in] config_file Name of the server configuration file. Use "" for default. (default: "server.cfg")
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NOT_INITIALIZED_ERROR, ::PST_ERROR_STATUS_TRACKER_ERROR.
 *
 * @see pst_tracker_destroy()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_init2(PstTracker* ctracker, const char path[], const char config_file[]);

/**
 * @brief This function initializes the tracker object.
 *
 * This function initializes the tracker object. Camera calibration will be checked. When calibration files for the connected tracker
 * can not be found, a message with download links will be printed to the command line. Note that an initialized tracker object should
 * be destroyed with pst_tracker_destroy() before the tracker object is freed.
 *
 * @note Ensure that a tracker object is initialized only once.
 *
 * @param[in] ctracker The tracker object to be initialized.
 * @param[in] path Path to the configuration directory to be used for storing camera calibrations, target database and configuration files. Use "" for default. (default: "%PROGRAMDATA%\PS-Tech\PST Iris" on Windows or ~/.pstiris on Linux)
 * @param[in] config_file Name of the server configuration file. Use "" for default. (default: "server.cfg")
 * @param[in] db_file Name of the tracking target database file. Use "" for default. (default: "models.db")
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NOT_INITIALIZED_ERROR, ::PST_ERROR_STATUS_TRACKER_ERROR.
 *
 * @see pst_tracker_destroy()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_init3(PstTracker* ctracker, const char path[], const char config_file[], const char db_file[]);

/**
 * @brief This function initializes the tracker object.
 *
 * This function initializes the tracker object. Camera calibration will be checked. When calibration files for the connected tracker
 * can not be found, a message with download links will be printed to the command line. Note that an initialized tracker object should
 * be destroyed with pst_tracker_destroy() before the tracker object is freed.
 *
 * @note Ensure that a tracker object is initialized only once.
 *
 * @param[in] ctracker The tracker object to be initialized.
 * @param[in] path Path to the configuration directory to be used for storing camera calibrations, target database and configuration files. Use "" for default. (default: "%PROGRAMDATA%\PS-Tech\PST Iris" on Windows or ~/.pstiris on Linux)
 * @param[in] config_file Name of the server configuration file. Use "" for default. (default: "server.cfg")
 * @param[in] db_file Name of the tracking target database file. Use "" for default. (default: "models.db")
 * @param[in] grabber_name Name of the grabber plugin to be used. Can contain absolute or relative path to xml config file. Set to "default" to use auto-detect. (default: "default")
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NOT_INITIALIZED_ERROR, ::PST_ERROR_STATUS_TRACKER_ERROR.
 *
 * @see pst_tracker_destroy()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_init4(PstTracker* ctracker, const char path[], const char config_file[], const char db_file[], const char grabber_name[]);

/**
 * Deallocates the state of ctracker.
 */
PST_C_EXPORT void pst_tracker_destroy(PstTracker* ctracker);

/**
 * @brief Load calibration information from a given path.
 *
 * Load the calibration information for each of the cameras in the connected PST Tracker from the specified local path. This function will overwrite 
 * any calibration information for the currently connected trackers available in the current configuration folder. Calibration information will only 
 * be loaded and placed in the configuration folder when calibration information is found and valid for all connected cameras.
 *
 * @param[in] path Path to the local directory where the calibration information for all connected cameras is stored.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_INVALID_DATA_ERROR, ::PST_ERROR_STATUS_NOT_FOUND_ERROR, ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_tracker_get_connected_camera_urls()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_load_calibration_from_local_path(const PstTracker* ctracker, const char path[]);

/**
 * @brief Get URLs of the calibration information for the connected PST Tracker.
 *
 * Retrieves and prints the URLs to download the calibration information for the cameras in the connected PST Tracker. First initialize 
 * the urls with pst_camera_urls_init(). Destroy urls with pst_camera_urls_destroy() before freeing it if necessary.
 *
 * @param[in] ctracker The tracker object for which to request camera urls.
 * @param[in] silent Set to true in order to disable printing of calibration information.
 * @param[out] urls Object containing the urls for the connected cameras.

 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_FAILED_TO_ALLOCATE_MEMORY_ERROR, ::PST_ERROR_STATUS_NULLPTR_ERROR.
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_get_connected_camera_urls(const PstTracker* ctracker, bool silent, PstCameraURLs* urls);

/**
 * @brief Check if cameras of the connected PST Tracker are calibrated.
 *
 * Check for calibration information of the cameras of the connected PST Tracker. When not all calibration information can be retrieved,
 * download urls for the uncalibrated cameras will be returned. Furthermore, a message will be shown at the command line specifying which
 * files are missing and where they can be downloaded. First initialize the urls with pst_camera_urls_init().
 * Destroy urls with pst_camera_urls_destroy() before freeing it if necessary.
 *
 * @param[in] ctracker The tracker object for which to request camera urls.
 * @param[in] silent Set to true in order to disable printing of missing calibration information. (default: false)
 * @param[out] urls Object containing the urls for the uncalibrated cameras.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_FAILED_TO_ALLOCATE_MEMORY_ERROR, ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_camera_urls_init()
 * @see pst_camera_urls_destroy()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_get_uncalibrated_camera_urls(const PstTracker* ctracker, bool silent, PstCameraURLs* urls);

/**
 * @brief Get version information of the SDK.
 *
 * Retrieves version information of the PST Server used by the SDK. This version can differ from the SDK version retrieved using pst_sdk_get_version().
 * Memory will be allocated for *version_string. Ensure that it is freed using pst_free() when it is no longer needed.
 *
 * @param[in] ctracker The tracker object for which to request camera urls.
 * @param[out] version_string Contains a c style string with the version information.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_FAILED_TO_ALLOCATE_MEMORY_ERROR, ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_sdk_get_version()
 * @see pst_free()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_alloc_and_get_version_info(const PstTracker* ctracker, char** version_string);

/**
 * @brief Get the path to the current configuration directory.
 *
 * Retrieves the full path to the configuration directory currently being used by the PST Tracker. The default path is "%PROGRAMDATA%\PS-Tech\PST Iris" on Windows or ~/.pstiris on Linux.
 * The configuration directory contains the target model database, server configuration, reference file and tracker calibration information. Memory will be allocated for *config_path. Ensure that
 * it is freed using pst_free() when it is no longer needed.
 *
 * @param[in] ctracker The tracker object for which to request camera urls.
 * @param[out] config_path Contains a c style string with the version information.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_FAILED_TO_ALLOCATE_MEMORY_ERROR, ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_free()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_alloc_and_get_config_path(const PstTracker* ctracker, char** config_path);

/**
 * @brief Retrieve the latest PstTrackerData available from the connected PstTracker.
 *
 * Retrieve the latest PstTrackerData available from the PstTracker after the server started. If the PstTracker has not been started with pst_tracker_start(), 
 * the PstTrackerData will be empty. Make sure to initialize the PstTrackerData with pst_trackerdata_init() before using this function. The PstTrackerData can be reused, 
 * after its contents has been destroyed using pst_trackerdata_destroy(). When the PstTrackerData is not needed anymore, use pst_trackerdata_destroy() 
 * to free the content before removal.
 *
 * @param[in] ctracker The tracker object.
 * @param[in] cdata The latest PstTrackerData available retrieved from the PstTracker.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR, ::PST_ERROR_STATUS_FAILED_TO_ALLOCATE_MEMORY_ERROR.
 *
 * @see PstTracker
 * @see PstTrackerData
 * @see pst_tracker_start()
 * @see pst_trackerdata_init()
 * @see pst_trackerdata_destroy()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_get_single_measurement(const PstTracker* ctracker, PstTrackerData* cdata);

/**
 * @brief Add a callback function for tracker data.
 *
 * Register a callback function to receive tracking information from the connected PST Tracker. The callback function will be executed every time
 * data from the PST Tracker arrives. Multiple callback functions can be registered, and they will be executed in order of registration.
 * Note that if pst_tracker_add_tracker_data_callback() is used to add a callback, this callback can only be removed with pst_tracker_remove_tracker_data_callback()
 *
 * @param[in] ctracker The tracker object for which the on_tracker_data callback function has to be added.
 * @param[in] on_tracker_data A function pointer to the callback that should be executed when data arrives.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see PstTrackerData
 * @see pst_tracker_add_tracker_callbacks()
 * @see pst_tracker_remove_tracker_data_callback()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_add_tracker_data_callback(PstTracker* ctracker, void (*on_tracker_data)(const PstTrackerData* , EPstErrorStatus));

/**
 * @brief Add a callback function for mode changes.
 *
 * Register a callback function to process changes to the tracker mode. The callback function will be executed every time 
 * there is a change of mode in the tracker. Multiple callback functions can be registered, and they will be executed in order of registration.
 * Note that if pst_tracker_add_tracker_mode_callback() is used to add a callback, this callback can only be removed with pst_tracker_remove_tracker_mode_callback()
 *
 * @param[in] ctracker The tracker object for which the on_tracker_mode callback function has to be added.
 * @param[in] on_tracker_mode A function pointer to the callback that should be executed when a change of tracker mode happens.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see EPstTrackerMode
 * @see pst_tracker_add_tracker_callbacks()
 * @see pst_tracker_remove_tracker_mode_callback()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_add_tracker_mode_callback(PstTracker* ctracker, void (*on_tracker_mode)(EPstTrackerMode));

/**
 * @brief Add callback functions for tracker data and mode changes.
 *
 * Register two callback functions: one for receiving tracking information from the connected PST Tracker and one for reporting tracker mode changes.
 * If only one callback function needs to be registered, NULL can be passed for the other callback. Alternatively, pst_tracker_add_tracker_data_callback() 
 * and pst_tracker_add_tracker_mode_callback() can be used to register a single callback function.
 * The on_tracker_data callback function will be executed every time data from the PST Tracker arrives. The on_tracker_mode callback function will be 
 * executed every time there is a change of mode in the tracker.
 * Multiple callback functions can be registered, and they will be executed in order of registration.
 * Note that if pst_tracker_add_tracker_callbacks() is used to add callbacks, these callbacks can only be removed by calling pst_tracker_remove_tracker_callbacks()
 * using the same pair of function pointers.
 *
 * @param[in] ctracker The tracker object for which the callback functions have to be added.
 * @param[in] on_tracker_data A function pointer to the callback that should be executed when data arrives.
 * @param[in] on_tracker_mode A function pointer to the callback that should be executed when a change of tracker mode happens.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see PstTrackerData
 * @see EPstTrackerMode
 * @see pst_tracker_add_tracker_data_callback()
 * @see pst_tracker_add_tracker_mode_callback()
 * @see pst_tracker_remove_tracker_callbacks()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_add_tracker_callbacks(PstTracker* ctracker, void (*on_tracker_data)(const PstTrackerData* , EPstErrorStatus), void (*on_tracker_mode)(EPstTrackerMode));

/**
 * @brief Remove a callback function for tracker data.
 *
 * Remove a callback function receiving tracking information from the connected PST Tracker.
 * Note that only callback functions added with pst_tracker_add_tracker_data_callback() can be removed with this function.
 *
 * @param[in] ctracker The tracker object for which the callback function has to be removed.
 * @param[in] on_tracker_data A function pointer to the callback that should be removed.
 *
 * @return ::PST_ERROR_STATUS_OK on success, ::PST_ERROR_STATUS_NULLPTR_ERROR if the tracker does not exist or ::PST_ERROR_STATUS_NOT_FOUND_ERROR if the callback function is not found.
 *
 * @see pst_tracker_add_tracker_data_callback()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_remove_tracker_data_callback(PstTracker* ctracker, void (*on_tracker_data)(const PstTrackerData* , EPstErrorStatus));

/**
 * @brief Remove a callback function for mode changes.
 *
 * Remove the supplied tracker mode change callback function from the given PST Tracker.
 * Note that only callback functions added with pst_tracker_add_tracker_mode_callback() can be removed with this function.
 *
 * @param[in] ctracker The tracker object for which the callback function has to be removed.
 * @param[in] on_tracker_mode A function pointer to the callback that should be removed.
 *
 * @return ::PST_ERROR_STATUS_OK on success, ::PST_ERROR_STATUS_NULLPTR_ERROR if the tracker does not exist or ::PST_ERROR_STATUS_NOT_FOUND_ERROR if the callback function is not found.
 *
 * @see pst_tracker_add_tracker_mode_callback()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_remove_tracker_mode_callback(PstTracker* ctracker, void (*on_tracker_mode)(EPstTrackerMode));

/**
 * @brief Remove callback functions for tracker data and mode changes.
 *
 * Remove callback functions receiving tracking information and changes of mode from the connected PST Tracker.
 * Note that only pairs of callback functions added with pst_tracker_add_tracker_callbacks() can be removed with this function using the same pairs of function pointers.
 *
 * @param[in] ctracker The tracker object for which the callback functions have to be removed.
 * @param[in] on_tracker_data A function pointer to the callback that should be removed.
 * @param[in] on_tracker_mode A function pointer to the mode change callback that should be removed.
 *
 * @return ::PST_ERROR_STATUS_OK on success, ::PST_ERROR_STATUS_NULLPTR_ERROR if the tracker does not exist or ::PST_ERROR_STATUS_NOT_FOUND_ERROR if the pair of callback functions is not found.
 *
 * @see pst_tracker_add_tracker_callbacks()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_remove_tracker_callbacks(PstTracker* ctracker, void (*on_tracker_data)(const PstTrackerData* , EPstErrorStatus), void (*on_tracker_mode)(EPstTrackerMode));

/**
 * @brief Start tracking.
 *
 * Start the tracking system and produce tracking results.
 *
 * @param[in] ctracker The tracker object that should be started.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_tracker_pause()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_start(PstTracker* ctracker);

/**
 * @brief Pause tracking
 *
 * Pause the tracking system, stop producing tracking results and enter low-power mode.
 * Tracking can be resumed with a subsequent call to pst_tracker_start().
 *
 * @param[in] ctracker The tracker object that should be paused.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_tracker_start()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_pause(PstTracker* ctracker);

/**
 * @brief Check if the tracker is running correctly.
 *
 * Perform a system check, checking if the PST Tracker has been initialized correctly and if the system is running properly.
 * In order to get continuous feedback on the status of the PST Tracker it is recommended to implement regular polling of this
 * function. This way, issues that will not cause the software to crash (e.g. a camera disconnect) can be handled.
 *
 * @param[in] ctracker The tracker object.
 *
 * @return Current system status code as StatusMessage enumeration member.
 *
 * @see EPstStatusMessage
 */
PST_C_EXPORT EPstStatusMessage pst_tracker_system_check(const PstTracker* ctracker);

/**
 * @brief Set tracker frame rate.
 *
 * Set the PST Tracker frame rate to the value available for the connected PST Tracker nearest to the supplied value.
 *
 * @param[in] ctracker The tracker object.
 * @param[in] fps The new frame rate to be set.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NOT_SUPPORTED_ERROR, ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_tracker_get_framerate()
 * @see pst_tracker_alloc_and_get_supported_framerates()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_set_framerate(PstTracker* ctracker, double fps);

/**
 * @brief Get current frame rate.
 *
 * Get the current frame rate as set on the PST Tracker.
 *
 * @param[in] ctracker The tracker object.
 * @param[out] fps Current frame rate in frames per second (fps).
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_tracker_set_framerate()
 * @see pst_tracker_alloc_and_get_supported_framerates()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_get_framerate(const PstTracker* ctracker, double* fps);

/**
 * @brief Get an array of available frame rates.
 *
 * Get an array containing all frame rates that are available for the currently connected PST Tracker.
 * The range of available frame rates differs for different types of PST Trackers. When using
 * pst_tracker_set_framerate() to set a new frame rate, the frame rate of the PST Tracker will be set to the
 * value provided by this function nearest to the value supplied to ::pst_tracker_set_framerate. Ensure that
 * *framerates is freed using pst_free() when it is no longer needed.
 *
 * @param[in] ctracker The tracker object.
 * @param[out] framerates Pointer to a pointer to float. Will contain an array after function succeeds.
 * @param[out] number_of_framerates The size of the array *framerates.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR, ::PST_ERROR_STATUS_FAILED_TO_ALLOCATE_MEMORY_ERROR.
 *
 * @see pst_tracker_set_framerate()
 * @see pst_tracker_get_framerate()
 * @see pst_free()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_alloc_and_get_supported_framerates(const PstTracker* ctracker, float** framerates, size_t* number_of_framerates);

/**
 * @brief Set the exposure time.
 *
 * Set the PST Tracker exposure time to the supplied time. This adjusts both the shutter speed of the PST Tracker and the time the
 * IR illumination panel will be turned on.
 *
 * @param[in] ctracker The tracker object.
 * @param[in] time Exposure time in seconds.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NOT_SUPPORTED_ERROR, ::PST_ERROR_STATUS_NULLPTR_ERROR, ::PST_ERROR_STATUS_OUT_OF_RANGE_ERROR.
 *
 * @see pst_tracker_get_exposure()
 * @see pst_tracker_get_exposure_range()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_set_exposure(PstTracker* ctracker, double time);

/**
 * @brief Get the current exposure time.
 *
 * Get the current exposure time as set on the PST Tracker.
 *
 * @param[in] ctracker The tracker object.
 * @param[out] exposure The current exposure time in seconds.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_tracker_set_exposure()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_get_exposure(const PstTracker* ctracker, double* exposure);

/**
 * @brief Get the allowed exposure range.
 *
 * Get the minimum and maximum exposure value that can be set with the current frame rate.
 * The maximum exposure value differs for different frame rates and PST Trackers.
 * In general, lower frame rates allow for higher exposures and higher frame rates have a
 * lower maximum exposure value. After changing the frame rate it is advised to check the
 * currently allowed range before changing exposure values.
 *
 * @param[in] ctracker The tracker object.
 * @param[out] min The minimum exposure value that can be set.
 * @param[out] max The maximum exposure value that can be set.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_tracker_get_exposure()
 * @see pst_tracker_set_exposure()
 * @see pst_tracker_set_framerate()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_get_exposure_range(const PstTracker* ctracker, double* min, double* max);

/**
 * @brief Enable filtering of the tracking results.
 *
 * Switch on filtering of the tracker results using a double exponential-based prediction filter (DESP).
 * This will result in smoother tracking with less noise. It can have a small impact on tracker accuracy.
 *
 * @note When the shared memory communication pipeline is used to connect the PST-Client application to the
 * SDK, filtering will be disabled upon starting the PST-Client application. To prevent multiple filter passes,
 * filtering should not be re-enabled until the PST-Client application is closed.
 *
 * @param[in] ctracker The tracker object.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_tracker_disable_filtering()
 * @see pst_tracker_set_position_filter()
 * @see pst_tracker_set_orientation_filter()
 * @see pst_tracker_enable_tremor_filter()
 * @see pst_sdk_enable_shared_memory()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_enable_filtering(PstTracker* ctracker);

/**
 * @brief Disable filtering of the tracking results.
 *
 * Switch off filtering of the tracking results using the double exponential-based prediction filter (DESP).
 *
 * @param[in] ctracker The tracker object.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_tracker_enable_filtering()
 * @see pst_tracker_disable_tremor_filter()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_disable_filtering(PstTracker* ctracker);

/**
 * @brief Set the strength of the position filter.
 *
 * Set the strength of the double exponential-based prediction filter (DESP) that filters the estimated positions of the tracked tracking targets.
 * The supplied filtering value should be between 0 (no filtering) and 0.99 (maximum filtering).
 *
 * @param[in] ctracker The tracker object.
 * @param[in] value Filtering strength to be used in range [0, 0.99]
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_tracker_enable_filtering()
 * @see pst_tracker_set_orientation_filter()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_set_position_filter(PstTracker* ctracker, double value);

/**
 * @brief Set the strength of the orientation filter.
 *
 * Set the strength of the double exponential-based prediction filter (DESP) that filters the estimated orientation of the tracked tracking targets.
 * The supplied filtering value should be between 0 (no filtering) and 0.99 (maximum filtering).
 *
 * @param[in] ctracker The tracker object.
 * @param[in] value Filtering strength to be used in range [0, 0.99]
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_tracker_enable_filtering()
 * @see pst_tracker_set_position_filter()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_set_orientation_filter(PstTracker* ctracker, double value);

/**
 * @brief Enable the tremor filter.
 *
 * Enables the tremor filter. This filter greatly reduces the noise levels of the estimated target pose. However, using the filter introduces latency
 * to the tracking results received and lowers tracking accuracy. This filter is mainly useful when using the PST Tracker in an interaction-type setting.
 * When the PST Tracker is used for measurement purposes, enabling the tremor filter is not recommended.
 *
 * @note When the shared memory communication pipeline is used to connect the PST-Client application to the
 * SDK, filtering will be disabled upon starting the PST-Client application. To prevent multiple filter passes,
 * filtering should not be re-enabled until the PST-Client application is closed.
 *
 * @param[in] ctracker The tracker object.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_tracker_disable_tremor_filter()
 * @see pst_tracker_enable_filtering()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_enable_tremor_filter(PstTracker* ctracker);

/**
 * @brief Disable the tremor filter.
 *
 * Disables the tremor filter. The tremor filter will no longer be used to filter the estimated target pose.
 *
 * @param[in] ctracker The tracker object.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_tracker_enable_tremor_filter()
 * @see pst_tracker_disable_filtering()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_disable_tremor_filter(PstTracker* ctracker);

/**
 * @brief Enable image transfer from the PST Tracker.
 *
 * Enable transferring grayscale image data from the PST Tracker. After enabling image transfer, initialize an ::PstImage
 * with pst_image_init() and use pst_tracker_get_pst_image() to receive captured images. When no calls are made to
 * pst_tracker_get_pst_image() for more than 4 seconds, image transfer will be automatically disabled. Note that for the
 * standard PST Iris and standard PST Base trackers, enabling image transfer will result in a reduced framerate of 30 fps.
 * The frame rate will be restored to the original setting after image transfer is disabled.
 *
 * @param[in] ctracker The tracker object.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see PstImage
 * @see pst_image_init()
 * @see pst_image_destroy()
 * @see pst_tracker_get_pst_image()
 * @see pst_tracker_disable_image_transfer()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_enable_image_transfer(PstTracker* ctracker);

/**
 * @brief Disable image transfer from the PST Tracker.
 *
 * Disable transferring grayscale image data from the PST Tracker. When using a standard PST Iris or PST Base tracker,
 * the frame rate of the tracker will be restored to the frame rate that was set before enabling image transfer.
 *
 * @param[in] ctracker The tracker object.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_tracker_enable_image_transfer()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_disable_image_transfer(PstTracker* ctracker);

/**
 * @brief Retrieve images from the connected PST Tracker.
 *
 * Retrieve images from the PST Tracker when image transfer has been enabled. When image transfer has not been enabled by
 * calling pst_tracker_enable_image_transfer() before calling this method, PstImage.images will be an empty array. Note that
 * image transfer will be automatically disabled when no call to this method is made for 4 seconds.
 *
 * Note that PstImage.images is a pointer to an array of (likely) more than 1 grayscale images. These pointers could be invalidated by the
 * next call of pst_tracker_get_pst_image(). If the images are still needed after this call, they should be deepcopied.
 *
 * Make sure to initialize the PstImage with pst_image_init() before using this function. Afterwards the PstImage can be reused.
 * This function will ensure that the data in the PstImage object is properly freed before new data is stored.
 * In case the image retrieval fails ::PST_ERROR_STATUS_IMAGE_RETRIEVAL_FAILED_ERROR is returned, but PstImage is left untouched.
 * Note that the pointers in the image.images array might still be invalidated.
 *
 * @param[in] ctracker The tracker object.
 * @param[out] image The image information retrieved from the PST Tracker.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR, ::PST_ERROR_STATUS_FAILED_TO_ALLOCATE_MEMORY_ERROR, ::PST_ERROR_STATUS_IMAGE_RETRIEVAL_FAILED_ERROR.
 *
 * @see PstImage
 * @see pst_image_init()
 * @see pst_image_destroy()
 * @see pst_tracker_enable_image_transfer()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_get_pst_image(const PstTracker* ctracker, PstImage* image);

/**
 * @brief Get TargetStatuses object containing all tracking targets and their status.
 *
 * Retrieves the list of all tracking targets defined in the tracking target database and returns them as a TargetStatuses object.
 * Make sure to free *statuses using pst_free() when the PstTargetStatus objects are no longer needed.
 *
 * @param[in] ctracker The tracker object.
 * @param[out] statuses Pointer to allocated array of PstTargetStatus objects.
 * @param[out] number_of_statuses Number of elements in statuses.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR, ::PST_ERROR_STATUS_FAILED_TO_ALLOCATE_MEMORY_ERROR.
 *
 * @see PstTargetStatus
 * @see pst_tracker_set_target_status()
 * @see pst_tracker_get_target_status()
 * @see pst_free()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_alloc_and_get_target_list(const PstTracker* ctracker, PstTargetStatus** statuses, size_t* number_of_statuses);

/**
 * @brief Get basic tracking target information.
 *
 * Retrieves the name, uuid and id for the requested tracking target.
 *
 * @param[in] ctracker The tracker object.
 * @param[in] name The name of the tracking target.
 * @param[out] target Target object containing the basic target information.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR, ::PST_ERROR_STATUS_NOT_FOUND_ERROR.
 *
 * @see PstTarget
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_get_target_info(const PstTracker* ctracker, const char*  name, PstTarget* target);

/**
 * @brief Set status of a single tracking Target.
 *
 * Sets the status of a single tracking Target to active (true) or inactive (false).
 *
 * @param[in] ctracker The tracker object.
 * @param[in] name The name of the tracking target to set the status of.
 * @param[in] set_active New status of the tracking target.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR, ::PST_ERROR_STATUS_NOT_FOUND_ERROR.
 *
 * @see PstTargetStatus
 * @see pst_tracker_get_target_status()
 * @see pst_tracker_alloc_and_get_target_list()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_set_target_status(PstTracker* ctracker, const char*  name, bool set_active);

/**
 * @brief Get status of a single tracking Target.
 *
 * Retrieves the status of a single tracking Target. Tracking targets can be either active (true) or inactive (false).
 *
 * @param[in] ctracker The tracker object.
 * @param[in] name The name of the tracking target to get the status of.
 * @param[out] status Status of the tracking Target.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR, ::PST_ERROR_STATUS_NOT_FOUND_ERROR.
 *
 * @see PstTargetStatus
 * @see pst_tracker_set_target_status()
 * @see pst_tracker_alloc_and_get_target_list()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_get_target_status(const PstTracker* ctracker, const char* name, bool* status);

/**
 * @brief Get 3D marker positions of stored tracking target.
 *
 * Gets a list of the 3D positions of the markers making up the specified tracking target. The returned positions are
 * relative to the tracking target origin.
 *
 * Make sure to initialize the PstTargetMarkers with pst_target_markers_init() before using this function.
 * Afterwards the PstTargetMarkers can be reused in this function. This function will ensure that the data in the
 * PstTargetMarkers object is properly freed before new data is stored. When the PstTargetMarkers object is no longer
 * needed, please use pst_target_markers_destroy() to free the object.
 *
 * @param[in] ctracker The tracker object.
 * @param[in] name Name of the tracking target to get the marker positions of.
 * @param[out] marker_list Contains a list of 3D marker positions of the target.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR, ::PST_ERROR_STATUS_NOT_FOUND_ERROR, ::PST_ERROR_STATUS_FAILED_TO_ALLOCATE_MEMORY_ERROR.
 *
 * @see PstTargetMarkers
 * @see PstTarget
 * @see pst_target_markers_init()
 * @see pst_target_markers_destroy()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_get_target_markers(const PstTracker* ctracker, const char* name, PstTargetMarkers* marker_list);

/**
 * @brief Set the id of the tracking target.
 *
 * This function changes the id of the tracking target. This id is mainly used when connecting to the PST Tracker through VRPN.
 *
 * @param[in] ctracker The tracker object.
 * @param[in] name The name of the tracking target to change the id of.
 * @param[in] id The new id of the tracking target.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR, ::PST_ERROR_STATUS_NOT_FOUND_ERROR.
 *
 * @see PstTarget
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_set_target_id(PstTracker* ctracker, const char* name, int id);

/**
 * @brief Set the reference system in which tracking results are reported
 *
 * Sets the reference system relative to which tracking results are reported. The reference system is defined by a homogeneous transformation matrix
 * (a row-major 4x4 matrix) representing a right-handed Cartesian coordinate system. The function checks if the rotation component of the supplied 
 * transformation matrix is orthonormal and if the supplied transformation matrix is homogeneous (i.e. the final row of the matrix is [0 0 0 1]). If 
 * either on of these conditions is not true, the function will return ::PST_ERROR_STATUS_TRACKER_ERROR. As an optional parameter, the reference 
 * system can be set relative to the current reference system. When this option is used, the current reference system is transformed using the 
 * supplied transformation matrix. Otherwise, the current reference system is replaced by the supplied reference system.
 *
 * @param[in] ctracker The tracker object.
 * @param[in] creference The transformation matrix for the new reference system.
 * @param[in] relative When set to true, the current transformation matrix is transformed using the supplied transformation matrix.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR, ::PST_ERROR_STATUS_TRACKER_ERROR.
 *
 * @see pst_tracker_get_reference()
 * @see pst_tracker_set_default_reference()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_set_reference(PstTracker* ctracker, const float creference[16], bool relative);

/**
 * @brief Reset the reference system to the default reference system.
 *
 * Resets the currently set reference system to the default reference system. The default system is defined by a 4x4 identity matrix.
 * This places the origin of the reference system 1 meter away from the PST Tracker. It is oriented such that the Z-axis points away
 * from the PST tracker and the X-axis is parallel to the PST tracker.
 *
 * @param[in] ctracker The tracker object.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_tracker_set_reference()
 * @see pst_tracker_get_reference()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_set_default_reference(PstTracker* ctracker);

/**
 * @brief Gets the transformation matrix for the current reference system.
 *
 * Tracking results are reported relative to a predefined right-handed Cartesian coordinate system, called the reference system.
 * The default reference system is located at 1 meter from the PST Tracker. It is oriented such that the Z-axis points away from
 * the PST tracker and the X-axis is parallel to the PST tracker. The transformation matrix defining the reference system is a
 * row-major 4x4 homogeneous transformation matrix.
 *
 * @param[in] ctracker The tracker object.
 * @param[out] creference The transformation matrix for the current reference system.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_tracker_set_reference()
 * @see pst_tracker_set_default_reference()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_get_reference(const PstTracker* ctracker, float creference[16]);

/**
 * @brief Enable shared memory communication layer.
 *
 * Enable a shared memory-based communication layer. Enabling shared memory will allow the PST-Client application to connect to the SDK.
 * This will allow receiving tracking data and setting parameters using the PST-Client application.
 * Note that a PST-Client that is connected in this way will not be updated when settings are changed through the SDK. This could cause
 * instabilities in the PST software. It is recommended to only use this function when setting up a PST Tracker (e.g. adding tracking
 * targets or retrieving calibration data). When using a tracker in normal operation, it is recommended to choose to use either the
 * PST-Client application or the SDK and not mix usage.
 *
 * @note When using shared memory on Windows the application needs to have elevated access (run as Administrator). When running without
 * elevated access the PST-Client will not be able to connect.
 *
 * @note When the shared memory communication pipeline is used to connect the PST-Client application to the
 * SDK, filtering will be disabled upon starting the PST-Client application. To prevent multiple filter passes,
 * filtering should not be re-enabled until the PST-Client application is closed.
 * @note A PstTracker must have been initialized before enabling shared memory.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NOT_INITIALIZED_ERROR.
 *
 * @see pst_sdk_disable_shared_memory()
 * @see pst_tracker_enable_filtering()
 * @see pst_tracker_enable_tremor_filter()
 */
PST_C_EXPORT EPstErrorStatus pst_sdk_enable_shared_memory();

/**
 * @brief Disable shared memory communication layer.
 *
 * Disable the shared memory-based communication layer. The PST-Client application will no longer be able to connect to the PST Tracker.
 *
 * @see pst_sdk_enable_shared_memory()
 */
PST_C_EXPORT void pst_sdk_disable_shared_memory();

/**
 * @brief Enable a REST Server using the HTTP protocol on a local area network.
 *
 * Enable a REST Server that uses the HTTP protocol to enable access to the PST Tracker over a local area network.
 * Parameters from the tracker can be accessed using GET requests. Tracking results and image can be streamed using an Event-stream based communication pipeline.
 * The state of the tracker and parameters can be set through POST calls.
 * If a REST server is aleady running, calling pst_sdk_enable_rest_server() will terminate the current server and restart it with the new parameters.
 * More information on how to use the REST server can be found in the \ref REST documentation.
 * If error result ::PST_ERROR_STATUS_HTTP_ERROR is returned, a possible cause could be an invalid IP address or port number.
 *
 * @note A PstTracker must have been initialized before enabling the REST server.
 *
 * @param[in] server_address The IP address on which the REST server should be accessible.
 * @param[in] server_port The port number on which the REST server should be accessible.
 * @param[in] event_stream_retry_timeout Timeout in milliseconds before an eventstream attempt to reconnect to the server automatically.
 *
 * @see pst_sdk_disable_rest_server()
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NOT_INITIALIZED_ERROR, ::PST_ERROR_STATUS_HTTP_ERROR.
 */
PST_C_EXPORT EPstErrorStatus pst_sdk_enable_rest_server(const char* server_address, const char* server_port, int event_stream_retry_timeout);

/**
 * @brief Disable the REST server communication layer.
 *
 * Disable the REST server communication layer. If a REST server is running, the connections will be closed and the REST server will be destroyed.
 * If no REST server is running a call to pst_sdk_disable_rest_server() does nothing.
 *
 * @see pst_sdk_enable_rest_server()
*/
PST_C_EXPORT void pst_sdk_disable_rest_server();

/**
 * @brief Remove a tracking target model from the target model database.
 *
 * Remove a tracking target model from the target model database.
 *
 * @param[in] ctracker The tracker object.
 * @param[in] model_name The name of the tracking target model to be removed.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR, ::PST_ERROR_STATUS_NOT_FOUND_ERROR.
 *
 * @see pst_tracker_import_json_model()
 * @see pst_tracker_export_json_model()
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_remove_target_model(PstTracker* ctracker, const char* model_name);

/**
 * Initializes PstCameraURLs.
 *
 * @param[in] urls Object to be initialized.
 */
PST_C_EXPORT void pst_camera_urls_init(PstCameraURLs* urls);

/**
 * Destroys PstCameraURLs.
 *
 * @param[in] urls Object to be destroyed.
 */
PST_C_EXPORT void pst_camera_urls_destroy(PstCameraURLs* urls);

/**
 * Initialization function for PstTrackerData
 *
 * @param[out] trackerdata Empty trackerdata object to be initialized.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_trackerdata_destroy()
 * @see pst_tracker_get_single_measurement()
 */
PST_C_EXPORT void pst_trackerdata_init(PstTrackerData* cdata);

/**
 * Destroy function for PstTrackerData
 */
PST_C_EXPORT void pst_trackerdata_destroy(PstTrackerData* cdata);

/**
 * Initialization function for PstImage
 *
 * @param[out] image Empty image object to be initialized.
 *
 * @return ::PST_ERROR_STATUS_OK on success or ::PST_ERROR_STATUS_NULLPTR_ERROR.
 *
 * @see pst_image_destroy()
 * @see pst_tracker_get_pst_image()
 */
PST_C_EXPORT void pst_image_init(PstImage* image);

/**
 * Destroy function for PstImage
 */
PST_C_EXPORT void pst_image_destroy(PstImage* image);

/**
 * Initialization function for PstTargetMarkers
 *
 * @param[in] marker_list Empty marker_list object to be initialized.
 *
 * @see pst_target_markers_destroy()
 * @see pst_tracker_get_target_markers()
 */
PST_C_EXPORT void pst_target_markers_init(PstTargetMarkers* marker_list);

/**
 * Destruction function for PstTargetMarkers
 *
 * @param[in] marker_list The marker_list object to be destroyed.
 *
 * @see pst_target_markers_init()
 * @see pst_tracker_get_target_markers()
 */
PST_C_EXPORT void pst_target_markers_destroy(PstTargetMarkers* marker_list);

/**
 * @brief Retrieve the SDK version string.
 *
 * Retrieves the SDK version information as a c style string. The SDK version can differ from the internal tracker version which can be retrieved using pst_tracker_alloc_and_get_version_info().
 *
 * @see pst_tracker_alloc_and_get_version_info()
 */
PST_C_EXPORT const char* pst_sdk_get_version();

/**
 * @brief Write plug-in status information to the standard C output stream stderr.
 *
 * Provides information about the status of the tracker plug-in. Information is written to the standard C output stream stderr. Useful when debugging.
 *
 * @note This should be called before creating any initializing a PstTracker with pst_tracker_init(). Calling this afterwards will have no effect.
 */
PST_C_EXPORT void pst_sdk_enable_logging();

/**
 * @brief Shutdown the tracking system, stopping tracking.
 *
 * Fully shutdown the tracking system, disconnecting from the PST Tracker.
 * When the shared memory interface or the REST server have been enabled these will be disabled as well.
 * In order to restart the tracker, a new PstTracker has to be initialized using pst_tracker_init().
 *
 * @see pst_tracker_init()
 * @see pst_sdk_disable_shared_memory()
 * @see pst_sdk_disable_rest_server()
 */
PST_C_EXPORT void pst_sdk_shutdown();

/**
 * @brief Free data allocated by the PST SDK
 *
 * This function should be called on any data that was allocated by the PST SDK and is no longer needed by the user.
 * Use this function instead of the general free() function to prevent memory issues.
 *
 * @param[in] data Pointer to the memory to be freed.
 *
 * @see pst_alloc_and_get_last_error_message()
 * @see pst_tracker_alloc_and_get_version_info()
 * @see pst_tracker_alloc_and_get_config_path()
 * @see pst_tracker_alloc_and_get_supported_framerates()
 * @see pst_tracker_alloc_and_get_target_list()
 */
PST_C_EXPORT void pst_free(void* data);

/**
 * @brief Import a tracking target model into the target model database.
 *
 * Import a JSON-formatted string representing a tracking target model into the target model database.
 * The JSON string should be formatted according to the instructions found in Appendix A of the PST Manual.
 * After successfully importing the model it can be tracked by the PST Tracker.
 *
 * @param[in] ctracker The tracker object.
 * @param[in] json_model A JSON-formatted string representing the tracking target model.
 * 
 * @return ::PST_ERROR_STATUS_OK on success, ::PST_ERROR_STATUS_NULLPTR_ERROR if the tracker does not exist or ::PST_ERROR_STATUS_NOT_FOUND_ERROR, PST_ERROR_STATUS_INVALID_DATA_ERROR, PST_ERROR_STATUS_ALREADY_EXISTS_ERROR, PST_ERROR_STATUS_JSON_PARSING_ERROR.
 *
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_import_json_model(const PstTracker* ctracker, const char* json_model);

/**
 * @brief Export a tracking target model as a JSON-formatted string.
 *
 * Export a tracking target model from the target model database as a JSON-formatted string.
 * The string will be formatted according to the specification found in Appendix A of the PST Manual.
 * Memory will be allocated for *model_string. Ensure that it is freed using pst_free() when it is no longer needed.
 *
 * @param[in] ctracker The tracker object.
 * @param[in] model_name The name of the target model to be exported.
 * @param[out] model_string Contains a c style string with the exported JSON model.
 *
 * @return ::PST_ERROR_STATUS_OK on success, ::PST_ERROR_STATUS_NULLPTR_ERROR if the tracker does not exist or ::PST_ERROR_STATUS_NOT_FOUND_ERROR.
 *
 */
PST_C_EXPORT EPstErrorStatus pst_tracker_export_json_model(const PstTracker* ctracker, const char* model_name, char** model_string);

/** @} */
#ifdef __cplusplus
}
#endif
