"""Tracking target example of the PST SDK

This example shows how to work with tracking targets using the PST SDK.
Note that at this moment tracking targets can not be trained or
imported using the PST SDK. In order to add new tracking targets, please
use the PST Client together with the pstech.pstsdk.tracker.Tracker.enable_shared_memory()
function, or use the stand-alone PST Server to configure the tracking targets.

Copyright PS-Tech B.V. All Rights Reserved.
"""
import context
import time
import sys
import pstech.pstsdk.tracker as pst
import pstech.pstsdk.errors as psterrors

# Control variable for main loop
running = True

# Number of data points to grab before application termination
max_samples = 100

# Global number of samples
samples = 0

"""Implementation of the pst.Listener class to receive tracking data and mode changes."""
class MyListener(pst.Listener):

    """Implementation of a tracker data callback function

    Implementation of a tracker data callback function. The on_tracker_data
    function receives the data as soon as it becomes available.

    Args:
        tracker_data: Object containing tracking information retrieved from tracker
        status_message: Status message reported by the tracker.

    See Also:
        pstech.pstdk.trackerdata.TrackerData
        pstech.pstsdk.errors.EStatusMessage
    """
    def on_tracker_data(self, tracker_data, status_message):
        # Do something with the tracker data
        global samples
        global running

        if samples >= max_samples:
            running = False

        for target_pose in tracker_data.targetlist:
            print("Detected " + target_pose.name)

        samples += 1

"""Helper function to register the exit handler with the application"""
def register_exit_handler():
    if sys.platform.startswith("linux"):
        import signal
        signal.signal(signal.SIGTERM, exit_handler)
        signal.signal(signal.SIGHUP, exit_handler)
        signal.signal(signal.SIGQUIT, exit_handler)
        signal.signal(signal.SIGINT, exit_handler)
    elif sys.platform.startswith("win"):
        import win32api
        win32api.SetConsoleCtrlHandler(exit_handler, True)

"""Implement the exit handler to shut-down the PST Tracker connection on application termination."""
def exit_handler(*args):
    global running
    pst.Tracker.shutdown()
    running = False
    return True

def main():
    if(len(sys.argv) < 2):
        print("\nConfiguration Error: A camera configuration file needs to be specified. This file can be found in the Redist folder of your installation. "
                "See the documentation of the Python bindings for more information.")
        exit(0)

    # Register exit_handler for proper shutdown
    register_exit_handler()

    try:
        # Use Context Manager to prevent improper Tracker shutdown on errors.
        # Create an instance of the Tracker object using the default configuration path and file names.
        with pst.Tracker("", "","", sys.argv[1]) as tracker:

            # Print version number of the tracker server being used.
            print("Running PST Server version " + tracker.get_version_info())

            # Create listener with callback functions for data and/or mode updates.
            listener = MyListener()
            
            # Register the listener object to the tracker server.
            tracker.add_tracker_listener(listener)

            print("Put the Reference card in front of the PST in order to see tracking results.\n")

            # Start the tracker server.
            tracker.start()

            # Perform a system check to see if the tracker server is running OK and print the result.
            print("System check: " + str(tracker.system_check()))

            # Retrieve the list of registered tracking targets and print their names and current status (active or not).
            targets = tracker.get_target_list()
            print("Found " + str(len(targets)) + " tracking targets:")
            for target_status in targets:
                print(target_status.name + "\t" + str(target_status.status))
            print("")

            # Enable the Reference target. Note that this will cause a TrackerError in case the Reference
            # target has not been created. The Reference target can be created using the PST Client.
            tracker.set_target_status("Reference", True)

            # Get the 3D marker positions making up the Reference device and display them.
            # Note that this will cause a TrackerError in case the Reference target has not been created.
            markers_list = tracker.get_target_markers("Reference")
            print("3D marker positions making up the Reference target:")
            for marker in markers_list:
                print("x: " + str(marker[0]) + "\ty: " + str(marker[1]) + "\tz: " + str(marker[2]))
            print("")

            # Main loop, wait for auto-termination.
            while running:
                time.sleep(0.1)

    except psterrors.TrackerError as err:
        # Catch TrackerError exceptions and print error messages.
        print(err.message)

if __name__ == "__main__":
    main()
