﻿// Copyright PS-Tech B.V. All Rights Reserved.

using System;
using System.Runtime.InteropServices;
using System.Threading;
using PSTech.Pstsdk;

namespace PSTech.Images
{
    /// <summary>Images example of the PST SDK</summary>
    /// <remarks>
    /// This example shows how to enable image transfer on the PST Tracker and how to use
    /// the PST SDK to retrieve images. Images are 8 bit grayscale and are stored as an
    /// Intptr array without memory alignment or padding.
    /// </remarks>
    public class Images
    {
        /*
        * Define handler functions required to ensure a clean shutdown of the PST Tracker when the
        * application is terminated.
        */
        [DllImport("Kernel32")]
        private static extern bool SetConsoleCtrlHandler(ConsoleEventHandler handler, bool add);

        private delegate bool ConsoleEventHandler(CtrlType sig);
        static ConsoleEventHandler ConsoleCtrlEventHandler;

        public enum CtrlType
        {
            CtrlCEvent = 0,
            CtrlBreakEvent = 1,
            CtrlCloseEvent = 2,
            CtrlLogOffEvent = 5,
            CtrlShutdownEvent = 6,
        }

        /// <summary>Define handler functions required to ensure a clean shutdown of the PST Tracker when the application is terminated.</summary>
        /// <param name="sig">Signal type</param>
        /// <returns>Signal handling status</returns>
        private static bool ConsoleHandler(CtrlType sig)
        {
            Exithandler((int)sig);
            return true;
        }

        // Control variable for main loop.
        static bool Running = true;

        // Number of data points to grab before application termination.
        static int NumberOfSamplesToGrab = 1000;
        static int SamplesGrabbed = 0;

        /// Implementation of the PSTech.Pstsdk.Tracker.TrackerListener class to receive tracking data.
        class MyTrackerListener : TrackerListener
        {
            /// <summary>Implementation of a tracker data callback function</summary>
            /// <remarks>
            /// The OnTrackerData() callback function receives the data as soon as it becomes
            /// available.
            /// </remarks>
            /// <param name="data">Object containing tracking information retrieved from tracker</param>
            /// <param name="status">Status message reported by the tracker.</param>
            public override void OnTrackerData(TrackerData data, ErrorStatus status)
            {
                if (SamplesGrabbed++ >= NumberOfSamplesToGrab)
                {
                    Running = false;
                }

                // Do something with the received data.
            }
        }

        /// <summary>
        /// Implement the exit handler to shut-down the PST Tracker connection on application termination.
        /// </summary>
        /// <param name="sig">Signal type.</param>
        static void Exithandler(int sig)
        {
            Tracker.Shutdown();
            Environment.Exit(0);
        }
        static int Main(string[] args)
        {
            // Register the exit handler with the application
            ConsoleCtrlEventHandler += new ConsoleEventHandler(ConsoleHandler);
            SetConsoleCtrlHandler(ConsoleCtrlEventHandler, true);

            // Implement error handling of PSTech.TrackerException exceptions to prevent
            // improper PST Tracker shutdown on errors.
            try
            {
                // Create an instance of the Tracker object using the default configuration path and file names.
                Tracker tracker = new Tracker();

                // Print version number of the tracker server being used.
                Console.Write("Running PST Server version " + tracker.GetVersionInfo() + "\n");

                //Create tracker listener with callback functions for data and/or mode updates.
                TrackerListener listener = new MyTrackerListener();

                // Register the TrackerListener object to the tracker server.
                tracker.AddTrackerListener(ref listener);

                // Start the Tracker server.
                tracker.StartTracker();

                // Perform a system check to see if the tracker server is running OK and print the result.
                Console.Write("System check: " + tracker.Systemcheck() + "\n\n");
                Console.Write("***************************\n\n");

                // Set the frame rate to 60 Hz.
                tracker.SetFramerate(60);
                Console.Write("Current frame rate: " + tracker.GetFramerate() + " Hz\n\n");

                // In order to start receiving images, enable image transfer. When image transfer is disabled,
                // the vector of images returned by GetImage will be empty.
                tracker.EnableImageTransfer();

                // The standard PST trackers will run at a reduced frame rate of 30 Hz when image transfer is enabled.
                // However, since this frame rate is temporary for as long as image transfer is enabled, that frame rate
                // will not be reported as the current frame rate.
                Console.Write("Enabled image transfer. Current frame rate: " + tracker.GetFramerate() + " Hz\n\n");
                Console.Write("***************************\n\n");

                // Try to capture 100 images.
                for (int i = 0; i < 100; ++i)
                {
                    // Try to get the last grabbed image.
                    // Note that enabling image transfer takes some time. While image transfer is being enabled,
                    // the images array in the PSTech.Pstsdk.Image object will be empty.
                    // Image object return array of Intptr to create images.

                    Pstsdk.Image images = tracker.GetImage();
                    Console.Write("Retrieval operation successful!\n");
                    Console.Write("Retrieved " + images.Images.Length + " image(s) of size: " + images.Width + " X " + images.Height + "\n\n");

                    // Do something with the image.

                    // Don't request images too fast, wait for around 1/60 seconds
                    Thread.Sleep(17);
                }

                // Wait for 5 seconds, since this is > 4 seconds, image transfer will be disabled automatically.
                Console.Write("Waiting 5 seconds for image transfer to be automatically disabled...\n\n");
                Thread.Sleep(5000);

                // Try to grab one image. Since image retrieval timed out, it should return an empty image vector.
                Pstsdk.Image image = tracker.GetImage();
                Console.Write("Retrieval operation successful!\n");
                Console.Write("Retrieved " + image.Images.Length + " image(s) of size: " + image.Width + " X " + image.Height + "\n\n");

                // Main loop, wait for auto-termination.
                while (Running)
                {
                    Thread.Sleep(100) ;
                }

            }
            catch (Exception e)
            {
                // Catch PSTech.TrackerException exceptions and print error messages.
                Console.Write(e + "\n");
            }
            finally
            {
                // Make sure that the connection to the PST Tracker is shut down properly.
                Tracker.Shutdown();
            }

            // Pause command line to see results.
            Console.Write("Press enter to continue...\n");
            Console.Read();

            return 0;
        }
    }
}
