"""Module containing all target related classes and functions.

Copyright PS-Tech B.V. All Rights Reserved.
"""

import ctypes as c
from .point import Point

class Target:
    """Tracking target information.

    This class contains basic information about tracking targets known to the PST Tracker.
    In order to be accessible, these targets have to be trained or imported using the PST-Client software.
    The targets are stored in the model database.

    Attributes:
        name: User given name of the tracking target as listed in the PST-Client software
        uuid: Unique identifier, automatically assigned to the tracking target
        id: User provided tracking id assigned to the tacking target as listed in the PST-Client software
    """
    def __init__(self, c_target):
        self.name = c_target.name.decode("UTF-8")
        self.uuid = c_target.uuid.decode("UTF-8")
        self.id = c_target.id

    def __eq__(self, other):
        if not isinstance(other, Target):
            return NotImplemented
        return  self.name == other.name and \
                self.uuid == other.uuid and \
                self.id == other.id

    ##@cond
    class _CTarget(c.Structure):
        _fields_= [
            ('name', c.c_char * 128),
            ('uuid', c.c_char * 37),
            ('id', c.c_int)
        ]
    ##@endcond

class TargetPose(Target):
    """Estimated tracking target pose.

    This class contains the estimated pose information and matched points of a tracking target.
    The pose is provided as a row-major 4x4 transformation matrix describing the pose of the tracking target in the coordinate system
    defined in the PST-Client application (see the 'Reference coordinate system' Section in the PST Manual).
    The matched points are provided as a list of 3D data points that have been detected by the tracker and are estimated be part of the Target.

    Attributes:
        pose: Tracking target pose as a flattened row-major 4x4 transformation matrix
        matched_points: point.Point list representing detected 3D markers matched to this tracking target

    See Also:
        target.Target
        point.Point
        trackerdata.TrackerData
    """

    def __init__(self, c_pose):
        super().__init__(c_pose.target)
        self.pose = c_pose.pose_matrix[:]
        self.matched_points = []
        for idx in range(c_pose.number_of_matched_points):
            c_point = c_pose.matched_points[idx]
            point = Point(c_point)
            self.matched_points.append(point)

    def __eq__(self, other):
        if not isinstance(other, TargetPose):
            return NotImplemented
        return  super().__eq__(other) and \
                self.pose == other.pose and \
                self.matched_points == other.matched_points

    ##@cond
    class _CTargetPose(c.Structure):
        _fields_ = [
            ('target', Target._CTarget),
            ('pose_matrix', c.c_float * 16),
            ('number_of_matched_points', c.c_size_t),
            ('matched_points', c.POINTER(Point._CPoint))
        ]
    ##@endcond

class TargetStatus(Target):
    """Tracking target status.

    This class provides status information about a tracking target.
    The status is either true (active), in which case the object will be tracked when seen by the PST Tracker,
    or false (inactive), in which case it will not be tracked by the PST Tracker.

    Attributes:
        status: Status of the tracking target (active or inactive) as listed in the PST-Client software

    See Also:
        target.Target
        tracker.Tracker.get_target_list
        tracker.Tracker.set_target_status
        tracker.Tracker.get_target_status
    """
    def __init__(self, c_status):
        super().__init__(c_status.target)
        self.status = c_status.status

    def __eq__(self, other):
        if not isinstance(other, TargetStatus):
            return NotImplemented
        return  super().__eq__(other) and \
                self.status == other.status
    ##@cond
    class _CTargetStatus(c.Structure):
        _fields_ = [
            ('target', Target._CTarget),
            ('status', c.c_bool)
        ]
    ##@endcond
