"""Module containing all image related classes and functions.

Note:
    If numpy is installed, this module will use a numpy array for the
    representation of a greyscale image.
    If numpy is not installed, this module will use a ctypes array for the
    representation of a greyscale image.

Copyright PS-Tech B.V. All Rights Reserved.
"""

import ctypes as c
import sys
import importlib.util

##@cond
NUMPY = None
# Check if numpy can be imported and import
spec = importlib.util.find_spec("numpy")
if spec is not None:
    NUMPY = importlib.util.module_from_spec(spec)
    sys.modules["numpy"] = NUMPY
    spec.loader.exec_module(NUMPY)
##@endcond

class Image:
    """Images retrieved from the tracker.

    This class contains grayscale images retrieved from the cameras inside the PST Tracker.

    Note:
        If numpy is installed, the greyscale image is represented as a numpy array with values of type numpy.ubyte and without memory alignment.
        If numpy is not installed, the greyscale image is represented as a ctypes array with values of type ctypes.c_ubyte and without memory alignment.
        This is done for optimization reasons.

    Attributes:
        width: Integer describing width of the images
        height: Integer describing height of the images
        images: List containing greyscale images. An image is an array without memory alignment and consists of values of either numpy.ubyte or ctypes.c_ubyte.

    See Also:
        tracker.Tracker.enable_image_transfer
        tracker.Tracker.disable_image_transfer
        tracker.Tracker.get_image
    """
    def __init__(self, c_images):
        self.width = c_images.width
        self.height = c_images.height
        self.images = []
        for idx in range(c_images.number_of_images):
            c_image = c_images.images[idx]
            array_type = c.POINTER(c.c_ubyte * (self.width * self.height))
            image = array_type.from_buffer(c_image)[0]
            # Only convert to numpy array if numpy is available
            if NUMPY is not None:
                image = NUMPY.frombuffer(image, NUMPY.ubyte)
            self.images.append(image)

    def __eq__(self, other):
        if not isinstance(other, Image):
            return NotImplemented
        return  self.width == other.width and \
                self.height == other.height and \
                self.images == other.images
    ##@cond
    class _CImage(c.Structure):
        _fields_ = [
            ('width', c.c_size_t),
            ('height', c.c_size_t),
            ('number_of_images', c.c_size_t),
            ('images', c.POINTER(c.POINTER(c.c_ubyte))),
        ]
    ##@endcond
