"""Minimal example of the PST SDK

This is a bare minimum example showing how to connect to the PST SDK and how to
register a listener to receive data.

In order to be able to run this example, the PST Tracker has to be initialized first.
This can be done by starting the PST-Server and the PST-Client application and making
sure the calibration files have been downloaded and a tracking target is available.
The tracking target can be the default Reference target or a newly trained or imported
target. For more information, please see the Initialization section of the PST SDK manual
or check the PST Manual.

Copyright PS-Tech B.V. All Rights Reserved.
"""
import context
import time
import sys
import pstech.pstsdk.tracker as pst
import pstech.pstsdk.errors as psterrors

"""Implementation of the pst.Listener class to receive tracking data and mode changes."""
class MyListener(pst.Listener):

    """Implementation of a tracker data callback function

    Implementation of a tracker data callback function. The on_tracker_data
    function receives the data as soon as it becomes available.

    Args:
        tracker_data: Object containing tracking information retrieved from tracker
        status_message: Status message reported by the tracker.

    See Also:
        pstech.pstdk.trackerdata.TrackerData
        pstech.pstsdk.errors.EStatusMessage
    """
    def on_tracker_data(self, tracker_data, status_message):
        # Do something with the tracker data
        pass

def main():
    if(len(sys.argv) < 2):
        print("\nConfiguration Error: A camera configuration file needs to be specified. This file can be found in the Redist folder of your installation. "
                "See the documentation of the Python bindings for more information.")
        exit(0)

    try:
        # Use Context Manager to prevent improper Tracker shutdown on errors.
        # Create an instance of the Tracker object using the default configuration path and file names.
        with pst.Tracker("", "","", sys.argv[1]) as tracker:

            # Create listener with callback functions for data and/or mode updates.
            listener = MyListener()
            
            # Register the listener object to the tracker server.
            tracker.add_tracker_listener(listener)

            # Start the tracker server.
            tracker.start()

            # Wait for 10 seconds, allowing for the detection of tracking targets.
            time.sleep(10)
    except psterrors.TrackerError as err:
        # Catch TrackerError and print error messages.
        print(err.message)

if __name__ == "__main__":
    main()
