"""Rest server example of the PST SDK

This example shows how to enable the REST server using the PST SDK.
The REST server enables network-based access to the PST Tracker using
the HTTP protocol. GET and POST requests can be made to the server to
send and receive data and change parameters. The REST interface offers
a programming language independent interface to the PST Tracker.
Besides accessing the REST server directly using a browser, a networking
library can be used to interface with the server programatically.

Copyright PS-Tech B.V. All Rights Reserved.
"""
import context
import time
import sys
import pstech.pstsdk.tracker as pst
import pstech.pstsdk.errors as psterrors

"""Helper function to register the exit handler with the application"""
def register_exit_handler():
    if sys.platform.startswith("linux"):
        import signal
        signal.signal(signal.SIGTERM, exit_handler)
        signal.signal(signal.SIGHUP, exit_handler)
        signal.signal(signal.SIGQUIT, exit_handler)
        signal.signal(signal.SIGINT, exit_handler)
    elif sys.platform.startswith("win"):
        import win32api
        win32api.SetConsoleCtrlHandler(exit_handler, True)

"""Implement the exit handler to shut-down the PST Tracker connection on application termination."""
def exit_handler(*args):
    pst.Tracker.disable_rest_server()
    pst.Tracker.shutdown()

def main():
    if(len(sys.argv) < 2):
        print("\nConfiguration Error: A camera configuration file needs to be specified. This file can be found in the Redist folder of your installation. "
                "See the documentation of the Python bindings for more information.")
        exit(0)

    # Register exit_handler for proper shutdown
    register_exit_handler()

    try:
        # Use Context Manager to prevent improper Tracker shutdown on errors.
        # Create an instance of the Tracker object using the default configuration path and file names.
        with pst.Tracker("", "","", sys.argv[1]) as tracker:
            # Start the tracker server.
            tracker.start()

            # Enable the REST server. To check if the REST server is started correctly, browse to
            # http://localhost:7278/PSTapi/SystemCheck
            # In order to use 127.0.0.1 as an address on Windows 7, execute the following command
            # on an elevated command prompt to allow communication to this address:
            # "netsh http add urlacl url=http://127.0.0.1:7278/ user=EVERYONE listen=yes delegate=no"
            tracker.enable_rest_server("localhost", "7278")

            print("PST REST server enabled. See the PST SDK Manual for example commands.")

            # Wait for one minute before terminating this application.
            time.sleep(60)

            # Disable the REST server.
            tracker.disable_rest_server()
    except psterrors.TrackerError as err:
        # Catch TrackerError and print error messages.
        print(err.message)

if __name__ == "__main__":
    main()
