/**
* images.cpp
* Copyright PS-Tech B.V. All Rights Reserved.
*
* This example shows how to enable image transfer on the PST Tracker and how to use
* the PST SDK to retrieve images. Images are 8 bit grayscale and are stored as an
* unsigned char* without memory alignment or padding.
*
*/

#ifdef WIN32
#include <windows.h>
#else
#include <csignal>
#endif

/*
* Define handler functions required to ensure a clean shutdown of the PST Tracker when the
* application is terminated.
*/

static void Exithandler(int sig);

#ifdef WIN32
BOOL WINAPI ConsoleHandler(DWORD CEvent)
{
    Exithandler(CEvent);
    return TRUE;
}
#endif

/* End of handler functions */

#include <iostream>
#include <thread>
#include <chrono>
#include <atomic>

#include "pstsdk_cpp.h"
#include "TrackerExceptions.h"
#include "PstStringIoStream.h"

/* Control variable for main loop */
static std::atomic<bool> running(true);

/* Number of data points to grab before application termination */
static const uint32_t numberOfSamplesToGrab = 1000;

/*
* Implementation of the PSTech::pstsdk::Listener class to receive tracking data.
* The OnTrackerData() callback function receives the data as soon as it becomes
* available.
*/
class MyListener : public PSTech::pstsdk::Listener
{
    virtual void OnTrackerData(const PSTech::pstsdk::TrackerData& td)
    {
        static uint32_t samplesGrabbed = 0;
        if (samplesGrabbed++ >= numberOfSamplesToGrab)
            running = false;

        // Do something with the received data.
    }
} listener;

/*
* Implement the exit handler to shut-down the PST Tracker connection on application termination.
*/
static void Exithandler(int sig)
{
    PSTech::pstsdk::Tracker::Shutdown();
    exit(0);
}

int main(int argc, char *argv[])
{
    // Register the exit handler with the application
#ifdef WIN32
    SetConsoleCtrlHandler((PHANDLER_ROUTINE)ConsoleHandler, TRUE);
#else
    signal(SIGTERM, Exithandler);
    signal(SIGKILL, Exithandler);
    signal(SIGQUIT, Exithandler);
    signal(SIGINT, Exithandler);
#endif

    // Implement error handling of PSTech::TrackerException exceptions to prevent 
    // improper PST Tracker shutdown on errors.
    try
    {
        // Create an instance of the Tracker object using the default configuration path and file names.
#ifdef WIN32
        PSTech::pstsdk::Tracker pst;
#else
        // On Linux, specify the type of grabber that needs to be used as the last parameter: 
        // "basler_ace" for PST HD or "basler_dart" for PST Pico
        PSTech::pstsdk::Tracker pst("","config.cfg","models.db",argv[1]);
#endif
        // Print version number of the tracker server being used.
        std::cout << "Running PST Server version " << pst.GetVersionInfo() << "\n";

        // Register the listener object to the tracker server.
        pst.AddTrackerListener(&listener);

        // Start the tracker server.
        pst.Start();

        // Perform a system check to see if the tracker server is running OK and print the result.
        std::cout << "System check: " << (int)pst.Systemcheck() << "\n\n";
        std::cout << "***************************\n\n";

        // Set the frame rate to 60 Hz.
        pst.SetFramerate(60);
        std::cout << "Current frame rate: " << pst.GetFramerate() << " Hz\n\n";

        // In order to start receiving images, enable image transfer. When image transfer is disabled, 
        // the vector of images returned by GetImage will be empty.
        pst.EnableImageTransfer();

        // The standard PST trackers will run at a reduced frame rate of 30 Hz when image transfer is enabled.
        // However, since this frame rate is temporary for as long as image transfer is enabled, that frame rate
        // will not be reported as the current frame rate.
        std::cout << "Enabled image transfer. Current frame rate: " << pst.GetFramerate() << " Hz\n\n";
        std::cout << "***************************\n\n";

        // Try to capture 100 images.
        for (int i = 0; i < 100; ++i)
        {
            // Get a reference to the last grabbed image in the PSTech::pstsdk::Image struct.
            // Note that enabling image transfer takes some time. While image transfer is being enabled,
            // the images vector in the PSTech::pstsdk::Image struct will be empty.
            PSTech::pstsdk::Image images;
            bool succes = pst.GetImage(images);
            std::cout << "Retrieval operation " << ((succes) ? "successful!\n" : "unsuccessful!\n\n");
            if (succes)
            {
                std::cout << "Retrieved " << images.images.size() << " image(s) of size: " << images.width << " X " << images.height << "\n\n";
                // Do something with the image.
            }
            // Don't request images too fast, wait for around 1/60 seconds
            std::this_thread::sleep_for(std::chrono::milliseconds(17));
        }

        // Wait for 5 seconds, since this is > 4 seconds, image transfer will be disabled automatically.
        std::cout << "Waiting 5 seconds for image transfer to be automatically disabled...\n\n";
        std::this_thread::sleep_for(std::chrono::milliseconds(5000));

        // Try to grab one image. Since image retrieval timed out, it should return an empty image vector.
        PSTech::pstsdk::Image images;
        bool succes = pst.GetImage(images);
        std::cout << "Retrieval operation " << ((succes) ? "successful!\n" : "unsuccessful!");
        if (succes)
        {
            std::cout << "Retrieved " << images.images.size() << " image(s) of size: " << images.width << " X " << images.height << "\n\n";
        }

        // Main loop, wait for auto-termination.
        while (running)
        {
            std::this_thread::sleep_for(std::chrono::milliseconds(100));
        }
    }
    catch (PSTech::TrackerException &e)
    {
        // Catch PSTech::TrackerException exceptions and print error messages.
        std::cout << e.full_description() << "\n";
    }

    // Make sure that the connection to the PST Tracker is shut down properly.
    PSTech::pstsdk::Tracker::Shutdown();

    // Pause command line to see results.
    std::cout << "Press enter to continue...\n";
    std::cin.get();

    return 0;
}