"""Module containing all tracker related classes and functions.

Copyright PS-Tech B.V. All Rights Reserved.
"""

import ctypes as c
from enum import IntEnum
from typing import List
from .errors import TrackerError
from .errors import _EErrorStatus
from .errors import EStatusMessage
from .image import Image
from .target import Target
from .target import TargetStatus
from .trackerdata import TrackerData
from ._capi import _CApi

##@cond
_on_tracker_data_callback_type = c.CFUNCTYPE(None, c.POINTER(TrackerData._CTrackerData), c.c_int)
_on_tracker_mode_callback_type = c.CFUNCTYPE(None, c.c_int)
_capi = _CApi()

def _check_error(status_code):
    error_status = _EErrorStatus(status_code)
    if error_status != _EErrorStatus.OK:
        c_string = c.c_char_p()
        _capi.pst_alloc_and_get_last_error_message(c.byref(c_string))
        err_msg = ""
        if c_string:
            err_msg = c_string.value.decode("UTF-8") #pylint: disable=no-member
        _capi.pst_free(c_string)
        error_status.raise_error(err_msg)
##@endcond


class ETrackerMode(IntEnum):
    """ Tracker mode enum class

    This enum class lists all modes that can be reported by the tracker.

    Attributes:
        MODE_LOWPOWER: Tracker is in lowpower mode waiting for activity to resume.
        MODE_SETUP: Tracker is in setup mode.
        MODE_TRACKING: Tracker is in tracking mode.
        MODE_TRAINING: Tracker is in training mode.
        MODE_DISCONNECT: Tracker has been disconnected.
        MODE_RECONNECT: Tracker has been reconnected.
        MODE_ERROR: Tracker has encountered an error.
        MODE_UNKNOWN: Tracker is in an unknown state.
    """
    MODE_LOWPOWER               = 0
    MODE_SETUP                  = 1
    MODE_TRACKING               = 2
    MODE_TRAINING               = 3
    MODE_DISCONNECT             = 4
    MODE_RECONNECT              = 5
    MODE_ERROR                  = 6
    MODE_UNKNOWN                = 7

def get_sdk_version():
    """Retrieve the SDK version string.

    Retrieves the SDK version information as a string. The SDK version can differ from the internal tracker version which can be retrieved using GetVersionInfo.

    Returns:
        The version of SDK.

    See Also:
        Tracker.get_version_info
    """
    buff = _capi.pst_sdk_get_version()
    return buff.decode('UTF-8')

def enable_logging():
    """Write plug-in status information to the standard error stream.

    Provides information about the status of the tracker plug-in. Information is written to the standard error stream. Useful when debugging.

    Note:
        This should be called before creating any Tracker objects. Calling this afterwards will have no effect.
    """
    _capi.pst_sdk_enable_logging()

class Listener:
    """Abstract listener class for receiving tracking information and tracking mode updates from the tracker.

    Extend this class and implement its methods in order to receive tracking information and tracking mode updates from the PST Tracker. Examples on how to use this class and how to get
    data and modes from the PST Tracker can be found in the Examples section.
    Alternatively, Tracker.get_single_measurement can be called to get the latest available tracking data.

    See Also:
        Tracker.add_tracker_listener
        Tracker.remove_tracker_listener
        Tracker.get_single_measurement
    """

    def on_tracker_data(self, tracker_data, status_message):
        """Callback function receiving tracking information from the tracker.

        Implement the on_tracker_data method to receive tracking information from the PST Tracker.

        Args:
            TrackerData: The object containing the detected tracking targets and 3D markers.
            EStatusMessage: Status reported by the tracker.

        See Also:
            trackerdata.TrackerData
            errors.EStatusMessage
        """

    def on_tracker_mode(self, mode):
        """Callback function receiving mode updates from the tracker.

        Implement the on_tracker_mode method to receive mode updates from the PST Tracker.
        This function enables responding to the different PST Tracker mode changes enumerated by ETrackerMode as soon as they occur.

        Args:
            ETrackerMode: Current tracker mode code as ETrackerMode enumeration member.

        See Also:
            tracker.ETrackerMode
        """

class Tracker:
    """Main PST SDK class implementing tracker communication.

    This class implements the main PST SDK functionality. Examples on how to use this class and how to get data from the PST Tracker
    can be found in the Examples section.
    """
    def __init__(self, path="", config_file="", db_file="", grabber_name=""):
        r"""Tracker constructor.

        Construct Tracker object and initialize connected PST tracker. Camera calibration information is checked during initialization.
        When calibration files for the connected tracker can not be found, a message with download links will be printed to the command line.

        Args:
            path: Path to the configuration directory to be used for storing camera calibrations, target database and configuration files.
                Use "" for default. (default: "%PROGRAMDATA%\PS-Tech\PST Iris" on Windows or ~/.pstiris on Linux)
            config_file: Name of the server configuration file. Use "" for default. (default: "server.cfg")
            db_file: Name of the tracking target database file. Use "" for default. (default: "models.db")
            grabber_name: Name of the grabber plugin to be used. Can contain absolute or relative path to xml config file. Set to "default" to use auto-detect. (default: "default")

        Raises:
            NotInitializedError: Unable to initialize Tracker
            TrackerError: Tracker was not initialized correctly

        See Also:
            Tracker.get_uncalibrated_camera_urls
            errors.NotInitializedError
            errors.TrackerError
        """
        self._listeners: List[Listener] = []
        self._on_tracker_data_cfunc = None
        self._on_tracker_mode_cfunc = None
        self._tracker = Tracker._CTracker()
        bpath = path.encode("UTF-8")
        bconfig = config_file.encode("UTF-8")
        bmodels = db_file.encode("UTF-8")
        bgrabber = grabber_name.encode("UTF-8")
        _check_error(_capi.pst_tracker_init4(c.byref(self._tracker), bpath, bconfig, bmodels, bgrabber))

    def __enter__(self):
        return self

    def __exit__(self, ex_type, ex_value, ex_traceback):
        _capi.pst_tracker_destroy(c.byref(self._tracker))
        return False

    def load_calibration_from_local_path(self, path):
        """Load calibration information from a given path.

        Load the calibration information for each of the cameras in the connected PST Tracker from the specified local path. This function will overwrite
        any calibration information for the currently connected trackers available in the current configuration folder. Calibration information will only
        be loaded and placed in the configuration folder when calibration information is found and valid for all connected cameras.

        Args:
            path: Path to the local directory where the calibration information for all connected cameras is stored.

        Raises:
            NotFoundError: Calibration information not found for one or more of the connected cameras.
            InvalidDataError: The provided data is not valid.

        See Also:
            Tracker.get_connected_camera_urls
            errors.NotFoundError
            errors.InvalidDataError
        """
        bpath = path.encode("UTF-8")
        _check_error(_capi.pst_tracker_load_calibration_from_local_path(c.byref(self._tracker), bpath))

    def get_connected_camera_urls(self, silent = False):
        """Get URLs of the calibration information for the connected PST Tracker

        Returns and prints the URLs to download the calibration information for the cameras in the connected PST Tracker.

        Args:
            silent: Set to True in order to disable printing of calibration information. (default: False)

        Returns:
            List of URLs specifying the download location of the calibration information. The size of the list is the number of connected cameras.
        """
        connected_cameras = Tracker._CCameraURLs()
        _capi.pst_camera_urls_init(c.byref(connected_cameras))
        try:
            _check_error(_capi.pst_tracker_get_connected_camera_urls(c.byref(self._tracker), silent, c.byref(connected_cameras)))
            urls = []
            for idx in range(connected_cameras.number_of_urls):
                url = connected_cameras.url_data[idx]
                urls.append(url.decode("UTF-8"))
            return urls
        finally:
            _capi.pst_camera_urls_destroy(c.byref(connected_cameras))

    def get_uncalibrated_camera_urls(self, silent = False):
        """Check if cameras of the connected PST Tracker are calibrated

        Check for calibration information of the cameras of the connected PST Tracker. When not all calibration information can be retrieved,
        download urls for the uncalibrated cameras will be returned. Furthermore, a message will be shown at the command line specifying which
        files are missing and where they can be downloaded.

        Args:
            silent: Set to True in order to disable printing of missing calibration information. (default: False)

        Returns:
            List of URLs specifying the download location of calibration information for uncalibrated cameras. The size of the list is the number of uncalibrated cameras.
        """
        uncalibrated_cameras = Tracker._CCameraURLs()
        _capi.pst_camera_urls_init(c.byref(uncalibrated_cameras))
        try:
            _check_error(_capi.pst_tracker_get_uncalibrated_camera_urls(c.byref(self._tracker), silent, c.byref(uncalibrated_cameras)))
            urls = []
            for idx in range(uncalibrated_cameras.number_of_urls):
                url = uncalibrated_cameras.url_data[idx]
                urls.append(url.decode("UTF-8"))
            return urls
        finally:
            _capi.pst_camera_urls_destroy(c.byref(uncalibrated_cameras))

    def get_version_info(self):
        """Get version information of the SDK.

        Retrieves version information of the PST Server used by the SDK.

        Returns:
            The version number of the PST Server.

        See Also:
            get_sdk_version
        """
        version_info = ""
        buff = c.c_char_p()
        try:
            _check_error(_capi.pst_tracker_alloc_and_get_version_info(c.byref(self._tracker), c.byref(buff)))
            if buff:
                version_info = buff.value.decode("UTF-8") #pylint: disable=no-member
        finally:
            _capi.pst_free(buff)

        return version_info

    def get_config_path(self):
        """Get the path to the current configuration directory.

        Retrieves the full path to the configuration directory currently being used by the PST Tracker. The default path is "%PROGRAMDATA%/PS-Tech/PST Iris" on Windows or ~/.pstiris on Linux.
        The configuration directory contains the target model database, server configuration, reference file and tracker calibration information.

        Returns:
            The full path to the configuration directory.
        """
        config_path = ""
        buff = c.c_char_p()
        try:
            _check_error(_capi.pst_tracker_alloc_and_get_config_path(c.byref(self._tracker), c.byref(buff)))
            if buff:
                config_path = buff.value.decode("UTF-8") #pylint: disable=no-member
        finally:
            _capi.pst_free(buff)
        return config_path

    ##@cond
    def _c_on_tracker_data_callback_function(self, tracker_data, status_message):
        data = TrackerData(tracker_data.contents)
        for listener in self._listeners:
            if listener.on_tracker_data is not None:
                listener.on_tracker_data(data, _EErrorStatus(status_message))
    ##@endcond

    ##@cond
    def _c_on_tracker_mode_callback_function(self, mode):
        for listener in self._listeners:
            if listener.on_tracker_mode is not None:
                listener.on_tracker_mode(ETrackerMode(mode))
    ##@endcond

    def get_single_measurement(self):
        """Retrieve latest TrackerData available from the connected PST Tracker.

        Retrieve the latest TrackerData available from the PST Tracker.
        If the PST Tracker has not been started with start(), the TrackerData will be empty.

        Returns:
            The latest TrackerData available retrieved from the PST Tracker.

        See Also:
            trackerdata.TrackerData
            Tracker.start
            Listener
        """
        c_trackerdata = TrackerData._CTrackerData()
        _capi.pst_trackerdata_init(c.byref(c_trackerdata))
        data = None
        try:
            _check_error(_capi.pst_tracker_get_single_measurement(c.byref(self._tracker), c.byref(c_trackerdata)))
            data = TrackerData(c_trackerdata)
        finally:
            _capi.pst_trackerdata_destroy(c.byref(c_trackerdata))
        return data

    def add_tracker_listener(self, listener: Listener):
        """Add a listener for receiving tracker data and tracking mode updates.

        Register a listener to receive tracking information and tracking modes from the connected PST Tracker. The use of this method requires an implementation of the Listener class to be made.

        Args:
            listener: Listener object implementing the callback functions to receive tracker data and/or tracker mode updates

        See Also:
            Listener
            trackerdata.TrackerData
            tracker.ETrackerMode
            Tracker.remove_tracker_listener
        """
        if not self._on_tracker_data_cfunc and not self._on_tracker_mode_cfunc:
            # Store cfunc to keep it from being deleted and to later delete it
            self._on_tracker_data_cfunc = _on_tracker_data_callback_type(self._c_on_tracker_data_callback_function)
            self._on_tracker_mode_cfunc = _on_tracker_mode_callback_type(self._c_on_tracker_mode_callback_function)
            try:
                _check_error(_capi.pst_tracker_add_tracker_listener(c.byref(self._tracker), self._on_tracker_data_cfunc, self._on_tracker_mode_cfunc))
            except TrackerError:
                # Rest cfunc and reraise
                self._on_tracker_data_cfunc = None
                self._on_tracker_mode_cfunc = None
                raise

        self._listeners.append(listener)

    def remove_tracker_listener(self, listener: Listener):
        """Remove a listener for receiving tracker data and tracking mode updates.

        Remove a listener receiving tracking information and tracking modes from the connected PST Tracker.

        Args:
            listener: Listener object that was previously added through Tracker.add_tracker_listener().

        See Also:
            Listener
            Tracker.add_tracker_listener
        """
        try:
            self._listeners.remove(listener)
            if len(self._listeners) == 0 and self._on_tracker_data_cfunc and self._on_tracker_mode_cfunc:
                try:
                    _check_error(_capi.pst_tracker_remove_tracker_listener(c.byref(self._tracker), self._on_tracker_data_cfunc, self._on_tracker_mode_cfunc))
                finally:
                    self._on_tracker_data_cfunc = None
                    self._on_tracker_mode_cfunc = None
        except ValueError:
            # Callback doesnt exist, so do nothing
            pass

    def start(self):
        """Start tracking.

        Start the tracking system and produce tracking results.

        See Also:
            Tracker.pause
        """
        _check_error(_capi.pst_tracker_start(c.byref(self._tracker)))

    def pause(self):
        """Pause tracking

        Pause the tracking system, stop producing tracking results and enter low-power mode.
        Tracking can be resumed with a subsequent call to start().

        See Also:
            Tracker.start
        """
        _check_error(_capi.pst_tracker_pause(c.byref(self._tracker)))

    def system_check(self):
        """Check if the tracker is running correctly.

        Perform a system check, checking if the PST Tracker has been initialized correctly and if the system is running properly.
        In order to get continuous feedback on the status of the PST Tracker it is recommended to implement regular polling of this
        function. This way, issues that will not cause the software to crash (e.g. a camera disconnect) can be handled.

        Returns:
            Current system status code as EStatusMessage enumeration member

        See Also:
            errors.EStatusMessage
        """
        return EStatusMessage(_capi.pst_tracker_system_check(c.byref(self._tracker)))

    def set_framerate(self, framerate: float):
        """Set tracker frame rate.

        Set the PST Tracker frame rate to the value available for the connected PST Tracker nearest to the supplied value.

        Args:
            framerate: The new frame rate to be set.

        Raises:
            NotSupportedError: Supplied frame rate could not be set on the Tracker.

        See Also:
            Tracker.get_framerate
            Tracker.get_supported_framerates
            errors.NotSupportedError
        """
        _check_error(_capi.pst_tracker_set_framerate(c.byref(self._tracker), c.c_double(framerate)))

    def get_framerate(self):
        """Get current frame rate.

        Get the current frame rate as set on the PST Tracker.

        Returns:
            Current frame rate in frames per second (fps).

        See Also:
            Tracker.set_framerate
            Tracker.get_supported_framerates
        """
        framerate = c.c_double(0.0)
        _check_error(_capi.pst_tracker_get_framerate(c.byref(self._tracker), c.byref(framerate)))
        return framerate.value

    def get_supported_framerates(self):
        """Get vector of available frame rates.

        Get a vector containing all frame rates that are available for the currently connected PST Tracker.
        The range of available frame rates differs for different types of PST Trackers. When using
        Tracker.set_framerate() to set a new frame rate, the frame rate of the PST Tracker will be set to the
        value provided by this function nearest to the value supplied to Tracker.set_framerate().

        Returns:
            A list of available frame rates.

        See Also:
            Tracker.set_framerate
            Tracker.get_framerate
        """
        ret_list = []
        framerates = c.POINTER(c.c_float)()
        try:
            num_framerates = c.c_size_t()
            _check_error(_capi.pst_tracker_alloc_and_get_supported_framerates(c.byref(self._tracker), c.byref(framerates), c.byref(num_framerates)))
            for idx in range(num_framerates.value):
                ret_list.append(framerates[idx])
        finally:
            _capi.pst_free(framerates)
        return ret_list

    def set_exposure(self, exposure):
        """Set the exposure time.

        Set the PST Tracker exposure time to the supplied time. This adjusts both the shutter speed of the PST Tracker and the time the
        IR illumination panel will be turned on.

        Args:
            exposure: Exposure time in seconds.

        Raises:
            OutOfRangeError: The value supplied is not within the allowed range.
            NotSupportedError: Exposure could not be set on the Tracker.

        See Also:
            Tracker.get_exposure
            Tracker.get_exposure_range
            errors.OutOfRangeError
            errors.NotSupportedError
        """
        _check_error(_capi.pst_tracker_set_exposure(c.byref(self._tracker), c.c_double(exposure)))

    def get_exposure(self):
        """Get the current exposure time.

        Get the current exposure time as set on the PST Tracker.

        Returns:
            The current exposure time in seconds.

        See Also:
            Tracker.set_exposure
        """
        exposure = c.c_double(0.0)
        _check_error(_capi.pst_tracker_get_exposure(c.byref(self._tracker), c.byref(exposure)))
        return exposure.value

    def get_exposure_range(self):
        """Get the allowed exposure range.

        Get the minimum and maximum exposure value that can be set with the current frame rate.
        The maximum exposure value differs for different frame rates and PST Trackers.
        In general, lower frame rates allow for higher exposures and higher frame rates have a
        lower maximum exposure value. After changing the frame rate it is advised to check the
        currently allowed range before changing exposure values.

        Returns:
            Tuple of minimum and maximum exposure value that can be set.

        See Also:
            Tracker.get_exposure
            Tracker.set_exposure
            Tracker.set_framerate
        """
        exp_min = c.c_double(0.0)
        exp_max = c.c_double(0.0)
        _check_error(_capi.pst_tracker_get_exposure_range(c.byref(self._tracker), c.byref(exp_min), c.byref(exp_max)))
        return exp_min.value, exp_max.value

    def enable_filtering(self):
        """Enable filtering of the tracking results.

        Switch on filtering of the tracker results using a double exponential-based prediction filter (DESP).
        This will result in smoother tracking with less noise. It can have a small impact on tracker accuracy.

        Note:
            When the shared memory communication pipeline is used to connect the PST-Client application to the
            SDK, filtering will be disabled upon starting the PST-Client application. To prevent multiple filter passes,
            filtering should not be re-enabled until the PST-Client application is closed.

        See Also:
            Tracker.disable_filtering
            Tracker.set_position_filter
            Tracker.set_orientation_filter
            Tracker.enable_tremor_filter
            Tracker.enable_shared_memory
        """
        _check_error(_capi.pst_tracker_enable_filtering(c.byref(self._tracker)))

    def disable_filtering(self):
        """Disable filtering of the tracking results.

        Switch off filtering of the tracking results using the double exponential-based prediction filter (DESP).

        See Also:
            Tracker.enable_filtering
            Tracker.disable_tremor_filter
        """
        _check_error(_capi.pst_tracker_disable_filtering(c.byref(self._tracker)))

    def set_position_filter(self, value):
        """Set the strength of the position filter.

        Set the strength of the double exponential-based prediction filter (DESP) that filters the estimated positions of the tracked tracking targets.
        The supplied filtering value should be between 0 (no filtering) and 0.99 (maximum filtering).

        Args:
            value: Filtering strength to be used in range [0, 0.99]

        See Also:
            Tracker.enable_filtering
            Tracker.set_orientation_filter
        """
        _check_error(_capi.pst_tracker_set_position_filter(c.byref(self._tracker), c.c_double(value)))

    def set_orientation_filter(self, value):
        """Set the strength of the orientation filter.

        Set the strength of the double exponential-based prediction filter (DESP) that filters the estimated orientation of the tracked tracking targets.
        The supplied filtering value should be between 0 (no filtering) and 0.99 (maximum filtering).

        Args:
            value: Filtering strength to be used in range [0, 0.99]

        See Also:
            Tracker.enable_filtering
            Tracker.set_position_filter
        """
        _check_error(_capi.pst_tracker_set_orientation_filter(c.byref(self._tracker), c.c_double(value)))

    def enable_tremor_filter(self):
        """Enable the tremor filter.

        Enables the tremor filter. This filter greatly reduces the noise levels of the estimated target pose. However, using the filter introduces latency
        to the tracking results received and lowers tracking accuracy. This filter is mainly useful when using the PST Tracker in an interaction-type setting.
        When the PST Tracker is used for measurement purposes, enabling the tremor filter is not recommended.

        Note:
            When the shared memory communication pipeline is used to connect the PST-Client application to the
            SDK, filtering will be disabled upon starting the PST-Client application. To prevent multiple filter passes,
            filtering should not be re-enabled until the PST-Client application is closed.

        See Also:
            Tracker.disable_tremor_filter
            Tracker.enable_filtering
        """
        _check_error(_capi.pst_tracker_enable_tremor_filter(c.byref(self._tracker)))

    def disable_tremor_filter(self):
        """Disable the tremor filter.

        Disables the tremor filter. The tremor filter will no longer be used to filter the estimated target pose.

        See Also:
            Tracker.enable_tremor_filter
            Tracker.disable_filtering
        """
        _check_error(_capi.pst_tracker_disable_tremor_filter(c.byref(self._tracker)))

    def enable_image_transfer(self):
        """Enable image transfer from the PST Tracker.

        Enable transferring grayscale image data from the PST Tracker. After enabling image transfer, call Tracker.get_image()
        in order to receive the captured images. When no calls are made to Tracker.get_image() for more than 4 seconds, image
        transfer will be automatically disabled. Note that for the standard PST Iris and standard PST Base trackers, enabling image
        transfer will result in a reduced framerate of 30 fps. The frame rate will be restored to the original setting after
        image transfer is disabled.

        See Also:
            Tracker.get_image
            Tracker.disable_image_transfer
        """
        _check_error(_capi.pst_tracker_enable_image_transfer(c.byref(self._tracker)))

    def disable_image_transfer(self):
        """Disable image transfer from the PST Tracker.

        Disable transferring grayscale image data from the PST Tracker. When using a standard PST Iris or PST Base tracker,
        the frame rate of the tracker will be restored to the frame rate that was set before enabling image transfer.

        See Also:
            Tracker.enable_image_transfer
        """
        _check_error(_capi.pst_tracker_disable_image_transfer(c.byref(self._tracker)))

    def get_image(self):
        """Retrieve images from the connected PST Tracker.

        Retrieve images from the PST Tracker when image transfer has been enabled. When image transfer has not been enabled by
        calling Tracker.enable_image_transfer() before calling this method, Image.images will be an empty list. Note that
        image transfer will be automatically disabled when no call to this method is made for 4 seconds.

        Returns:
            The image information retrieved from the PST Tracker.

        See Also:
            image.Image
            Tracker.enable_image_transfer
        """
        c_image = Image._CImage()
        _capi.pst_image_init(c.byref(c_image))
        img = None
        try:
            _check_error(_capi.pst_tracker_get_pst_image(c.byref(self._tracker), c.byref(c_image)))
            img = Image(c_image)
        finally:
            _capi.pst_image_destroy(c.byref(c_image))
        return img

    def get_target_list(self):
        """Get TargetStatuses object containing all tracking targets and their status.

        Retrieves the list of all tracking targets defined in the tracking target database and returns them as a list of TargetStatus objects.

        Returns:
            List of TargetStatus objects containing the information and status of all available tracking Targets.

        See Also:
            target.TargetStatus
            Tracker.set_target_status
            Tracker.get_target_status
        """
        target_status_list = []
        c_target_status_list = c.POINTER(TargetStatus._CTargetStatus)()
        try:
            number_of_statuses = c.c_size_t()
            _check_error(_capi.pst_tracker_alloc_and_get_target_list(c.byref(self._tracker), c.byref(c_target_status_list), c.byref(number_of_statuses)))
            for idx in range(number_of_statuses.value):
                target_status_list.append(TargetStatus(c_target_status_list[idx]))
        finally:
            _capi.pst_free(c_target_status_list)
        return target_status_list

    def get_target_info(self, name):
        """Get basic tracking target information.

        Retrieves the name, uuid and id for the requested tracking target.

        Args:
            name: The name of the tracking target.

        Returns:
            Target object containing the basic target information.

        Raises:
            NotFoundError: Target name could not be found.

        See Also:
            target.Target
            errors.NotFoundError
        """
        target = Target._CTarget()
        bname = name.encode("UTF-8")
        _check_error(_capi.pst_tracker_get_target_info(c.byref(self._tracker), bname, c.byref(target)))
        return Target(target)

    def set_target_status(self, target_name, active):
        """Set status of a single tracking Target.

        Sets the status of a single tracking Target to active (true) or inactive (false).

        Args:
            target_name: The name of the tracking Target to set the status of.
            active: New status of the tracking Target.

        Raises:
            NotFoundError: Target name could not be found.

        See Also:
            target.TargetStatus
            Tracker.get_target_status
            Tracker.get_target_list
            errors.NotFoundError
        """
        bname = target_name.encode("UTF-8")
        _check_error(_capi.pst_tracker_set_target_status(c.byref(self._tracker), bname, active))

    def get_target_status(self, name):
        """Get status of a single tracking Target.

        Retrieves the status of a single tracking Target. Tracking targets can be either active (true) or inactive (false).

        Args:
            name: The name of the tracking Target to get the status of.

        Returns:
            Status of the tracking Target.

        Raises:
            NotFoundError: Target name could not be found.

        See Also:
            target.TargetStatus
            Tracker.set_target_status
            Tracker.get_target_list
            errors.NotFoundError
        """
        status = c.c_bool()
        bname = name.encode("UTF-8")
        _check_error(_capi.pst_tracker_get_target_status(c.byref(self._tracker), bname, c.byref(status)))
        return status.value

    def get_target_markers(self, target_name):
        """Get 3D marker positions of stored tracking Target.

        Gets a list of the 3D positions of the markers making up the specified tracking Target. The returned positions are
        relative to the tracking Target origin.

        Args:
            target_name: Name of the tracking Target to get the marker positions of.

        Returns:
            A list of tuples representing 3D marker positions.

        Raises:
            NotFoundError: Target name could not be found.

        See Also:
            target.Target
            errors.NotFoundError
        """
        c_markers = Tracker._CTargetMarkers()
        bname = target_name.encode("UTF-8")
        _capi.pst_target_markers_init(c.byref(c_markers))
        markers = []

        try:
            _check_error(_capi.pst_tracker_get_target_markers(c.byref(self._tracker), bname, c.byref(c_markers)))
            for idx in range(c_markers.number_of_markers):
                coordinates = (c_markers.markers[idx][0], c_markers.markers[idx][1], c_markers.markers[idx][2])
                markers.append(coordinates)
        finally:
            _capi.pst_target_markers_destroy(c.byref(c_markers))

        return markers

    def set_target_id(self, name, target_id):
        """Set the id of the tracking target.

        This function changes the id of the tracking target. This id is mainly used when connecting to the PST Tracker through VRPN.

        Args:
            name: The name of the tracking target to change the id of.
            target_id: The new id of the tracking target.

        Raises:
            NotFoundError: Target name could not be found.

        See Also:
            target.Target
            errors.NotFoundError
        """
        bname = name.encode("UTF-8")
        _check_error(_capi.pst_tracker_set_target_id(c.byref(self._tracker), bname, target_id))

    def set_reference(self, reference, relative = False):
        """Set the reference system in which tracking results are reported

        Sets the reference system relative to which tracking results are reported. The reference system is defined by a homogeneous transformation matrix
        (a row-major 4x4 matrix) representing a right-handed Cartesian coordinate system. The function checks if the rotation component of the supplied
        transformation matrix is orthonormal and if the supplied transformation matrix is homogeneous (i.e. the final row of the matrix is [0 0 0 1]). If
        either one of these conditions is not true, a TrackerError is thrown. As an optional parameter, the reference system can be set relative
        to the current reference system. When this option is used, the current reference system is transformed using the supplied transformation matrix.
        Otherwise, the current reference system is replaced by the supplied reference system.

        Args:
            reference: The transformation matrix for the new reference system.
            relative: When set to true, the current transformation matrix is transformed using the supplied transformation matrix.

        Raises:
            TrackerError: The supplied reference matrix is either non-homogeneous or the rotation component is non-orthonormal.

        See Also:
            Tracker.get_reference
            Tracker.set_default_reference
            errors.TrackerError
        """
        arr = (c.c_float * 16)(*reference)
        _check_error(_capi.pst_tracker_set_reference(c.byref(self._tracker), arr, relative))

    def set_default_reference(self):
        """Reset the reference system to the default reference system.

        Resets the currently set reference system to the default reference system. The default system is defined by a 4x4 identity matrix.
        This places the origin of the reference system 1 meter away from the PST Tracker. It is oriented such that the Z-axis points away
        from the PST tracker and the X-axis is parallel to the PST tracker.

        See Also:
            Tracker.set_reference
            Tracker.get_reference
        """
        _check_error(_capi.pst_tracker_set_default_reference(c.byref(self._tracker)))

    def get_reference(self):
        """Gets the transformation matrix for the current reference system.

        Tracking results are reported relative to a predefined right-handed Cartesian coordinate system, called the reference system.
        The default reference system is located at 1 meter from the PST Tracker. It is oriented such that the Z-axis points away from
        the PST tracker and the X-axis is parallel to the PST tracker. The transformation matrix defining the reference system is a
        row-major 4x4 homogeneous transformation matrix.

        Returns:
            The transformation matrix for the current reference system.

        See Also:
            Tracker.set_reference
            Tracker.set_default_reference
        """
        creference = (c.c_float * 16)()
        _check_error(_capi.pst_tracker_get_reference(c.byref(self._tracker), creference))
        return creference[:] #convert c_float array to a python list

    @staticmethod
    def enable_shared_memory():
        """Enable shared memory communication layer.

        Enable a shared memory-based communication layer. Enabling shared memory will allow the PST-Client application to connect to the SDK.
        This will allow receiving tracking data and setting parameters using the PST-Client application.
        Note that a PST-Client that is connected in this way will not be updated when settings are changed through the SDK. This could cause
        instabilities in the PST software. It is recommended to only use this function when setting up a PST Tracker (e.g. adding tracking
        targets or retrieving calibration data). When using a tracker in normal operation, it is recommended to choose to use either the
        PST-Client application or the SDK and not mix usage.

        Note:
            When using shared memory on Windows the application needs to have elevated access (run as Administrator). When running without
            elevated access the PST-Client will not be able to connect.

        Note:
            When the shared memory communication pipeline is used to connect the PST-Client application to the
            SDK, filtering will be disabled upon starting the PST-Client application. To prevent multiple filter passes,
            filtering should not be re-enabled until the PST-Client application is closed.

        Note:
            A Tracker object must have been initialized before enabling shared memory.

        Raises:
            NotInitializedError: A Tracker object must have been initialized before enabling shared memory.

        See Also:
            Tracker.disable_shared_memory
            Tracker.enable_filtering
            Tracker.enable_tremor_filter
            errors.NotInitializedError
        """
        _check_error(_capi.pst_sdk_enable_shared_memory())

    @staticmethod
    def disable_shared_memory():
        """Disable shared memory communication layer.

        Disable the shared memory-based communication layer. The PST-Client application will no longer be able to connect to the PST Tracker.

        See Also:
            Tracker.enable_shared_memory
        """
        _capi.pst_sdk_disable_shared_memory()

    @staticmethod
    def enable_rest_server(address = "localhost", port = "7278", event_stream_retry_timeout = 3000):
        """Enable a REST Server using the HTTP protocol on a local area network.

        Enable a REST Server that uses the HTTP protocol to enable access to the PST Tracker over a local area network.
        Parameters from the tracker can be accessed using GET requests. Tracking results and image can be streamed using an Event-stream based communication pipeline.
        The state of the tracker and parameters can be set through POST calls.
        If a REST server is aleady running, calling Tracker.enable_rest_server() will terminate the current server and restart it with the new parameters.
        More information on how to use the REST server can be found in the REST documentation.

        Note:
            A Tracker object must have been initialized before enabling shared memory.

        Args:
            address: The IP address on which the REST server should be accessible (default: localhost).
            port: The port number on which the REST server should be accessible (default:7278).
            event_stream_retry_timeout: Timeout in milliseconds before an eventstream attempt to reconnect to the server automatically (default:3000).

        Raises:
            HTTPError: The REST server failed to initialize properly. A possible cause could be an invalid IP address or port number.
            NotInitializedError: A Tracker object must have been initialized before enabling the REST server.

        See Also:
            Tracker.disable_rest_server
            errors.HTTPError
            errors.NotInitializedError
        """
        baddress = address.encode("UTF-8")
        bport = port.encode("UTF-8")
        _check_error(_capi.pst_sdk_enable_rest_server(baddress, bport, event_stream_retry_timeout))

    @staticmethod
    def disable_rest_server():
        """Disable the REST server communication layer.

        Disable the REST server communication layer. If a REST server is running, the connections will be closed and the REST server will be destroyed.
        If no REST server is running a call to Tracker.disable_rest_server() does nothing.

        See Also:
            Tracker.enable_rest_server
        """
        _capi.pst_sdk_disable_rest_server()

    @staticmethod
    def shutdown():

        """Shutdown the tracking system, stopping tracking.

        Fully shutdown the tracking system, disconnecting from the PST Tracker.
        When the shared memory interface or the REST server have been enabled these will be disabled as well.
        In order to restart the tracker, a new instance of the Tracker class has to be created.

        See Also:
            Tracker.__init__
            Tracker.disable_shared_memory
            Tracker.disable_rest_server
        """
        _capi.pst_sdk_shutdown()

    def import_json_model(self, json_model):
        """Import a tracking target model into the target model database.

        Import a JSON-formatted string representing a tracking target model into the target model database.
        The JSON string should be formatted according to the instructions found in Appendix A of the PST Manual.
        After successfully importing the model it can be tracked by the PST Tracker.

        Args:
            json_model: A JSON-formatted string representing the tracking target model.

        Raises:
            InvalidDataError: Data provided did not match the requirements.
            AlreadyExistsError: The request could not be executed because a model with the same name already exists in the database.
            JSONError: The JSON string could not be parsed correctly.

        See Also:
            Tracker.export_json_model
        """
        _check_error(_capi.pst_tracker_import_json_model(c.byref(self._tracker), json_model.encode("UTF-8")))

    def export_json_model(self, model_name):
        """Export a tracking target model as a JSON-formatted string.

        Export a tracking target model from the target model database as a JSON-formatted string.
        The string will be formatted according to the specification found in Appendix A of the PST Manual.

        Args:
            model_name: The name of the target model to be exported.

        Return:
            A JSON-formatted string representing the requested tracking target model.

        Raises:
            NotFoundError: The requested model was not found in the database.

        See Also:
            Tracker.import_json_model
        """
        model_string = ""
        buff = c.c_char_p()
        try:
            _check_error(_capi.pst_tracker_export_json_model(c.byref(self._tracker), model_name.encode("UTF-8"), c.byref(buff)))
            if buff:
                model_string = buff.value.decode("UTF-8") #pylint: disable=no-member
        finally:
            _capi.pst_free(buff)

        return model_string

    def remove_target_model(self, model_name):
        """Remove a tracking target model from the target model database.

        Remove a tracking target model from the target model database.

        Args:
            model_name: The name of the tracking target model to be removed.

        Raises:
            NotFoundError: The tracking target model to be removed was not found in the target model database.

        See Also:
            Tracker.import_json_model
            Tracker.export_json_model
        """
        _check_error(_capi.pst_tracker_remove_target_model(c.byref(self._tracker), model_name))

    ##@cond
    class _CTracker(c.Structure):
        _fields_ = [
            ('tracker', c.POINTER(c.c_void_p)),
        ]

    class _CCameraURLs(c.Structure):
        _fields_ = [('url_data', c.POINTER(c.c_char_p)),
                    ('number_of_urls', c.c_size_t)
                ]

    class _CTargetMarkers(c.Structure):
        _fields_ = [
            ('markers', c.POINTER(c.c_float * 3)),
            ('number_of_markers', c.c_size_t)
        ]
    ##@endcond
