// Copyright PS-Tech B.V. All Rights Reserved.

#pragma once

#include "PstBase.h"

#include <cstddef>

namespace PSTech
{
namespace Utils
{
    /**
     * @brief Basic vector class, cloning std::vector<T>.
     * 
     * This class can safely be passed across the DLL boundary.
     * 
     * PstVector<T> has been defined for the following types:
     * - float
     * - unsigned char*
     * - PstString
     * - PstArray<float, 3>
     * - pstsdk::Target
     * - pstsdk::TargetStatus
     * - pstsdk::TargetPose
     * - pstsdk::Point
     */
    template<typename T>
    class PST_EXPORT PstVector
    {
    public:
        typedef T value_type;
        typedef size_t size_type;
        typedef T* pointer;
        typedef const T* const_pointer;
        typedef T& reference;
        typedef const T& const_reference;
        typedef T* iterator;
        typedef const T* const_iterator;

        /** Default constructor creates empty PstVector of size 0 */
        PstVector();

        /** Construct a PstVector of size size, using the default constructor of T */
        explicit PstVector(size_type size);

        /** Construct a PstVector of size size and filling it with copies of data */
        PstVector(size_type size, const_reference data);

        /** Copy constructor */
        PstVector(const PstVector& vector);
        
        /** Move constructor */
        PstVector(PstVector&& vector) noexcept;

        /** Destructor */
        ~PstVector();

        /** Assignment operator */
        PstVector& operator=(const PstVector& vector);
        
        /** Move assignment operator */
        PstVector& operator=(PstVector&& vector) noexcept;

        /** Swaps the contents of the Pstvector with vector. */
        void swap(PstVector& vector) noexcept;

        /**
         * Get the current size of the PstVector
         * @note This returns the number of initialized elements in the PstVector, not the amount of allocated memory.
         * @see PstVector::capacity()
         */
        size_type size() const;

        /**
         * Get the number of elements for which memory has been allocated.
         * @note This is not the number of initialized elements in the PstVector.
         * @see PstVector::size()
         */
        size_type capacity() const;

        /** Returns true if the PstVector is empty. */
        bool empty() const;

        /**
         * Allocate space for a PstVector of size `size'.
         * This function ensures that the PstVector has enough memory allocated to contain `size' element, allocating more memory if needed.
         * @note PstVector::reserve() can only increase the amount of allocated memory.
         * @see PstVector::resize()
         */
        void reserve(size_type size);

        /**
         * Allocate memory for a PstVector of size `size' and initialize its elements using the defualt constructor of T.
         * @see PstVector::resize(size_type size, const_reference data)
         * @see PstVector::reserve()
         */
        void resize(size_type size);

        /**
         * Allocate memory for a PstVector of size `size' and initialize its elements using copies of `data'.
         * @see PstVector::resize(size_type size)
         * @see PstVector::reserve()
         */
        void resize(size_type size, const_reference data);

        /**
         * Destroy all elements of the PstVector and set its size to 0.
         * @note This does not deallocate the memory allocated for this PstVector.
         */
        void clear();

        /**
         * Fill the PstVector with `count' copies of `data'.
         * Resize the vector if `count' differs from the current PstVector size.
         */
        void assign(size_type count, const_reference data);

        /**
         * Add a copy of `data' to the back of the PstVector, increasing its size by one.
         * If not enough memory has been allocated to store the extra element, double the Pstvector size.
         */
        void push_back(const_reference data);

        /**
         * Move `data' to the back of the PstVector, increasing its size by one.
         * If not enough memory has been allocated to store the extra element, double the Pstvector size.
         */
        void push_back(T&& data);

        /**
         * Destroy the last element from the back of the PstVector, decreasing its size by one.
         */
        void pop_back();

        /**
         * Get a const reference to the item stored at position `ndex' in the PstVector with bounds check.
         * If `index' is larger than the size of the PstVector, a PSTech::OutOfRangeException is thrown.
         * @exception PSTech::OutOfRangeException
         */
        const_reference at(size_type index) const;

        /**
         * Get a reference to the item stored at position `ndex' in the PstVector with bounds check.
         * If `index' is larger than the size of the PstVector, a PSTech::OutOfRangeException is thrown.
         * @exception PSTech::OutOfRangeException
         */
        reference at(size_type index);

        /** Get a const reference to the first element in the PstVector. */
        const_reference front() const;

        /** Get a reference to the first element in the PstVector. */
        reference front();

        /** Get a const reference to the last element in the PstVector. */
        const_reference back() const;

        /** Get a reference to the last element in the PstVector. */
        reference back();

        /** Get a const iterator to the beginning of the PstVector. */
        const_iterator cbegin() const;
        
        /** Get a const iterator to the beginning of the PstVector. */
        const_iterator begin() const;

        /** Get an iterator to the beginning of the PstVector. */
        iterator begin();

        /** Get a const iterator to the end of the PstVector. */
        const_iterator cend() const;

        /** Get a const iterator to the end of the PstVector. */
        const_iterator end() const;

        /** Get an iterator to the end of the PstVector. */
        iterator end();

        /** Get a const reference to the item stored at position `ndex' in the PstVector. */
        const_reference operator [] (size_type index) const;

        /** Get a reference to the item stored at position `ndex' in the PstVector. */
        reference operator [] (size_type index);

    private:
        /** Simple basic allocator class to allocate and construct vector items. */
        class Allocator
        {
        public:
            /** Allocate but don't initialize num elements of type T */
            typename PstVector<T>::pointer allocate(typename PstVector<T>::size_type num);

            /** Initialize elements of allocated storage p using its default constructor */            
            void construct(typename PstVector<T>::pointer p);

            /** Initialize elements of allocated storage p using constructor with arguments arg */
            template<class... A>
            void construct(typename PstVector<T>::pointer p, A&... arg);

            /** Initialize elements of allocated storage p using the move constructor */
            void construct(typename PstVector<T>::pointer p, T&& data);

            /* Destroy elements of initialized storage p */
            void destroy(typename PstVector<T>::pointer p);

            /* Deallocate storage p of deleted elements */
            void deallocate(typename PstVector<T>::pointer p);
        };

        size_type m_size;
        size_type m_allocated;
        pointer m_array;
        Allocator m_allocator;

        void resize_internal(size_type size, const_pointer data);

        void check_bounds(size_type index) const;
    };
}
}
