// Copyright PS-Tech B.V. All Rights Reserved.

#pragma once

#include "PstString.h"
#include "PstVector.h"
#include "PstArray.h"

namespace PSTech
{
namespace pstsdk
{
    /**
     * @addtogroup pstsdk
     * @{
     */
     
    /**
     * @brief Images retrieved from the tracker.
     *
     * This struct contains pointers to grayscale images retrieved from the cameras inside the PST Tracker.
     *
     * @see Tracker::EnableImageTransfer()
     * @see Tracker::DisableImageTransfer()
     * @see Tracker::GetImage()
     */
    struct Image
    {
        unsigned int                     width;  /**< Image width in pixels */
        unsigned int                     height; /**< Image height in pixels */
        Utils::PstVector<unsigned char*> images; /**< Vector of pointers to retrieved grayscale images for each camera */
    };

    /**
     * @brief Tracking target information.
     *
     * This struct contains basic information about tracking targets known to the PST Tracker.
     * In order to be accessible, these targets have to be trained or imported using the PST-Client software.
     * The targets are stored in the model database.
     */
    struct Target
    {
        Utils::PstString                 name;   /**< User given name of the tracking target as listed in the PST-Client software */
        Utils::PstString                 uuid;   /**< Unique identifier, automatically assigned to the tracking target */
        int                              id;     /**< User provided tracking id assigned to the tacking target as listed in the PST-Client software */
    };

    /**
     * @brief Vector of Target objects.
     *
     * A vector containing tracking Target objects.
     *
     * @see Target
     */
    typedef Utils::PstVector<Target> Targets;

    /**
     * @brief Tracking target status.
     *
     * This struct provides status information about a tracking target.
     * The status is either true (active), in which case the object will be tracked when seen by the PST Tracker,
     * or false (inactive), in which case it will not be tracked by the PST Tracker.
     *
     * @see Target
     * @see Tracker::GetTargetList()
     * @see Tracker::SetTargetStatus()
     * @see Tracker::GetTargetStatus()
     */
    struct TargetStatus : public Target
    {
        bool                             status; /**< Status of the tracking target (active or inactive) as listed in the PST-Client software */
    };

    /**
     * @brief Vector of TargetStatus objects.
     *
     * A vector containing TargetStatus objects containing information about the status of tracking Targets.
     *
     * @see TargetStatus
     */
    typedef Utils::PstVector<TargetStatus> TargetStatuses;

    /**
     * @brief Estimated 3D marker location.
     *
     * This struct contains information about individual 3D markers that have been detected by the PST Tracker.
     * For a single frame, a Point is uniquely identified by its id.
     * Note that this id is not guaranteed to be consistent over time and can change between frames.
     *
     * @see TrackerData
     * @see Listener
     */
    struct Point
    {
        int                              id;         /**< Marker identifier */
        Utils::PstArray<float, 3>        position;   /**< 3D marker location */
    };

    /**
     * @brief Vector of Point objects.
     *
     * A vector containing Point objects.
     *
     * @see Point
     */
    typedef Utils::PstVector<Point> Points;

    /**
     * @brief Estimated tracking target pose and detected 3D markers.
     *
     * This struct contains the estimated pose information of a tracking target and detected 3D markers matched.
     * The pose is provided as a row-major 4x4 transformation matrix describing the pose of the tracking target in the coordinate system
     * defined by the Reference system that has been set (see the 'Reference coordinate system' Section in the PST Manual).
     * The 3D positions of the detected markers determined to be part of the target are provided as a Points vector.
     *
     * @see Target
     * @see TrackerData
     * @see Listener
     * @see Tracker::SetReference()
     */
    struct TargetPose : public Target
    {
        Utils::PstArray<float, 16>          pose;              /**< Tracking target pose as a row-major 4x4 transformation matrix */
        Points                              matched_markers;   /**< Vector of detected 3D markers matched to this tracking target */
    };

    /**
     * @brief Vector of TargetPose objects.
     *
     * A vector containing TargetPose objects containing information about the pose of tracking Targets.
     *
     * @see TargetPose
     */
    typedef Utils::PstVector<TargetPose> TargetPoses;

    /**
     * @brief Vector of 3D marker positions of a tracking Target.
     *
     * A vector of 3D marker positions of a tracking Target.
     *
     * @see Target
     */
    typedef Utils::PstVector<Utils::PstArray<float, 3>> MarkerList;

    /**
     * @brief Tracking information retrieved from tracker.
     *
     * This struct contains all tracking information recorded at a certain time instance. It provides a list of all
     * active tracking targets that have been detected by the PST Tracker, as well as a list of all 3D data points
     * that have been detected by the tracker and are estimated not to be part of any active tracking target.
     *
     * @see TargetPose
     * @see Point
     * @see Listener
     * @see Tracker::GetSingleMeasurement()
     */
    struct TrackerData
    {
        double                      timestamp;  /**< Timestamp of the moment the cameras captured the data. The timestamp uses the system clock provided in seconds since system boot (Windows) or Epoch (Linux). */
        int                         seqnumber;  /**< Incremental data recording id */
        TargetPoses                 targetlist; /**< Vector of detected tracking targets */
        Points                      datapoints; /**< Vector of detected 3D markers not matched to a tracking target */
    };

    /**
     * @brief Tracker status messages enum class.
     *
     * This enum class lists all status messages that can be reported by the tracker.
     *
     * @see Tracker::Systemcheck()
     */
    enum class StatusMessage
    {
        OK,                         /**< The system is running OK */
        NOT_INITIALIZED,            /**< Tracker failed to initialize correctly */
        DISCONNECTED,               /**< The grabber has been disconnected */
        ERR_GENERAL,                /**< Unspecified grabber error */
        ERR_TIMEOUT,                /**< Grabber timeout error */
        ERR_NOCAMS_FOUND,           /**< Grabber could not detect any cameras */
        ERR_NOTENOUGHTCAMS_FOUND,   /**< Grabber could not detect sufficient cameras */
        ERR_INITERROR,              /**< Grabber did not initialize correctly */
        ERR_CANNOT_START_CAMS,      /**< Grabber could not start cameras */
        ERR_CANNOT_SETUP_CAMS,      /**< Grabber failed setting up cameras */
    };

    /**
     * @brief Tracker mode enum class.
     *
     * This enum class lists all modes that can be reported by the tracker.
     *
     * @see Listener()
     */
    enum class TrackerMode
    {
        LOWPOWER,              /**< Tracker is in lowpower mode waiting for activity to resume */
        SETUP,                 /**< Tracker is in setup mode */
        TRACKING,              /**< Tracker is in tracking mode */
        TRAINING,              /**< Tracker is in training mode */
        DISCONNECT,            /**< Tracker has been disconnected */
        RECONNECT,             /**< Tracker has been reconnected */
        TRACKER_ERROR,         /**< Tracker has detected an error */
        UNKNOWN,               /**< Tracker is in an unknown state */
    };

    /** @} */

}
}
