"""Reference example of the PST SDK

This example shows how to use the PST SDK to adjust the PST Tracker reference system.
The reference system defines the Cartesian coordinate system in which tracking results
are reported. The example shows how to set the reference system by supplying a 4x4
homogeneous transformation matrix. It also shows how to check if the reference system
was set successfully.

Copyright PS-Tech B.V. All Rights Reserved.
"""
import context
import time
import sys
import pstech.pstsdk.tracker as pst
import pstech.pstsdk.errors as psterrors

#global variables for example
running = True
max_samples = 1000
samples = 0

"""Helper function for printing """
def print_matrix(matrix):
    for y in range(4):
        for x in range(4):
            print(str(matrix[x + y * 4]), end="\t")
        print("")
    print("\n")

# Control variable for main loop
running = True

# Number of data points to grab before application termination
max_samples = 100

# Global number of samples
samples = 0

"""Implementation of the pst.Listener class to receive tracking data and mode changes."""
class MyListener(pst.Listener):

    """Implementation of a tracker data callback function

    Implementation of a tracker data callback function. The on_tracker_data
    function receives the data as soon as it becomes available.

    Args:
        tracker_data: Object containing tracking information retrieved from tracker
        status_message: Status message reported by the tracker.

    See Also:
        pstech.pstdk.trackerdata.TrackerData
        pstech.pstsdk.errors.EStatusMessage
    """
    def on_tracker_data(self, tracker_data, status_message):
        global samples
        global running

        if samples >= max_samples:
            running = False

        samples += 1
        # Do something here with the received data

"""Helper function to register the exit handler with the application"""
def register_exit_handler():
    if sys.platform.startswith("linux"):
        import signal
        signal.signal(signal.SIGTERM, exit_handler)
        signal.signal(signal.SIGHUP, exit_handler)
        signal.signal(signal.SIGQUIT, exit_handler)
        signal.signal(signal.SIGINT, exit_handler)
    elif sys.platform.startswith("win"):
        import win32api
        win32api.SetConsoleCtrlHandler(exit_handler, True)

"""Implement the exit handler to shut-down the PST Tracker connection on application termination."""
def exit_handler(*args):
    global running
    pst.Tracker.shutdown()
    running = False
    return True

def main():
    if(len(sys.argv) < 2):
        print("\nConfiguration Error: A camera configuration file needs to be specified. This file can be found in the Redist folder of your installation. "
                "See the documentation of the Python bindings for more information.")
        exit(0)

    # Register exit_handler for proper shutdown
    register_exit_handler()

    try:
        # Use Context Manager to prevent improper Tracker shutdown on errors.
        # Create an instance of the Tracker object using the default configuration path and file names.
        with pst.Tracker("", "","", sys.argv[1]) as tracker:
            # Print version number of the tracker server being used.
            print("Running PST Server version " + tracker.get_version_info())

            # Create listener with callback functions for data and/or mode updates.
            listener = MyListener()
            
            # Register the listener object to the tracker server.
            tracker.add_tracker_listener(listener)

            # Start the tracker server.
            tracker.start()

            # Perform a system check to see if the tracker server is running OK and print the result.
            print("System check: " + str(tracker.system_check()))

            # Set the frame rate to 30 Hz.
            tracker.set_framerate(30)

            # Print the new frame rate to see if it was set correctly. Note that for PST HD and Pico
            # trackers the frame rate actually being set can differ from the value provided to Tracker.set_framerate().
            print("Frame rate set to: " + str(tracker.get_framerate()))
            print("***************************\n")

            # Get the transformation matrix for the current reference system.
            print("Current reference system transformation matrix:")
            print_matrix(tracker.get_reference())

            # Define new reference system transformation matrix rotating the reference system by
            # 90 degrees around the X-axis and 180 degrees around the Y-axis. Then translate the
            # origin of the reference system by 0.1 m in the X direction, -0.5 m in the Y direction
            # and 0.5 m in the Z direction.
            reference = [-1.0, 0.0, 0.0, 0.1,
                        0.0, 0.0, 1.0,-0.5,
                        0.0, 1.0, 0.0, 0.5,
                        0.0, 0.0, 0.0, 1.0]

            tracker.set_reference(reference)
            print("New reference system transformation matrix:")
            print_matrix(tracker.get_reference())


            # Trying to set the reference using a non-orthonormal transformation matrix (this should fail).
            non_orthonormal_reference = [-1.0, 1.0, 0.0, 0.1,
                                        0.0, 0.0,-1.0,-0.5,
                                        0.0,-1.0, 0.0, 0.5,
                                        0.0, 0.0, 0.0, 1.0
                                    ]
            try:
                tracker.set_reference(non_orthonormal_reference)
                print("Reference input incorrectly applied!")
            except psterrors.TrackerError as err:
                # This should fail because the supplied matrix is non-orthonormal
                print("Reference input correctly ignored: %s" % err.message)
            print("New reference system after applying non-orthonormal transformation:")
            print_matrix(tracker.get_reference())

            # Adjust the reference system by applying a relative transformation. The new reference system will have
            # the X-axis pointing outward from the tracker and the Y-axis parallel to the tracker front.
            relative_reference = [ 0.0,-1.0, 0.0, 0.5,
                                1.0, 0.0, 0.0, 0.4,
                                0.0, 0.0, 1.0, 0.0,
                                0.0, 0.0, 0.0, 1.0
                                ]
            tracker.set_reference(relative_reference, True)
            print("New reference system after applying relative transformation:")
            print_matrix(tracker.get_reference())
            print("***************************\n")

            # Reset reference system to default (origin at 1 m from center of the PST Tracker,
            # Z-axis pointing outward from the PST Tracker).
            tracker.set_default_reference()
            print("Reset default reference system:")
            print_matrix(tracker.get_reference())

            while running:
                time.sleep(0.1)

    except psterrors.TrackerError as err:
        # Catch TrackerError and print error messages.
        print(err.message)

if __name__ == "__main__":
    main()
