"""Images example of the PST SDK

This example shows how to enable image transfer on the PST Tracker and how to use
the PST SDK to retrieve images. Images are 8 bit grayscale and are stored as an
unsigned byte array without memory alignment or padding.

Copyright PS-Tech B.V. All Rights Reserved.
"""
import context
import time
import sys
import pstech.pstsdk.tracker as pst
import pstech.pstsdk.errors as psterrors

# Control variable for main loop
running = True

# Number of data points to grab before application termination
max_samples = 100

# Global number of samples
samples = 0

"""Implementation of the pst.Listener class to receive tracking data."""
class MyListener(pst.Listener):

    """Implementation of a tracker data callback function

    Implementation of a tracker data callback function. The on_tracker_data
    function receives the data as soon as it becomes available.

    Args:
        tracker_data: Object containing tracking information retrieved from tracker
        status_message: Status message reported by the tracker.

    See Also:
        pstech.pstdk.trackerdata.TrackerData
        pstech.pstsdk.errors.EStatusMessage
    """
    def on_tracker_data(self, tracker_data, status_message):
        global samples
        global running

        if samples >= max_samples:
            running = False

        samples += 1
        # Do something here with the received data

"""Helper function to register the exit handler with the application"""
def register_exit_handler():
    if sys.platform.startswith("linux"):
        import signal
        signal.signal(signal.SIGTERM, exit_handler)
        signal.signal(signal.SIGHUP, exit_handler)
        signal.signal(signal.SIGQUIT, exit_handler)
        signal.signal(signal.SIGINT, exit_handler)
    elif sys.platform.startswith("win"):
        import win32api
        win32api.SetConsoleCtrlHandler(exit_handler, True)

"""Implement the exit handler to shut-down the PST Tracker connection on application termination."""
def exit_handler(*args):
    global running
    pst.Tracker.shutdown()
    running = False
    return True

def main():
    if(len(sys.argv) < 2):
        print("\nConfiguration Error: A camera configuration file needs to be specified. This file can be found in the Redist folder of your installation. "
                "See the documentation of the Python bindings for more information.")
        exit(0)

    # Register exit_handler for proper shutdown
    register_exit_handler()

    try:
        # Use Context Manager to prevent improper Tracker shutdown on errors.
        # Create an instance of the Tracker object using the default configuration path and file names.
        with pst.Tracker("", "","", sys.argv[1]) as tracker:

            # Print version number of the tracker server being used.
            print("Running PST Server version " + tracker.get_version_info())

            # Create listener with callback functions for data and/or mode updates.
            listener = MyListener()
            
            # Register the listener object to the tracker server.
            tracker.add_tracker_listener(listener)

            # Start the tracker server.
            tracker.start()

            # Perform a system check to see if the tracker server is running OK and print the result.
            print("System check: " + str(tracker.system_check()))
            print("***************************\n")

            # Set the frame rate to 60 Hz.
            tracker.set_framerate(60)
            print("Current frame rate: " + str(tracker.get_framerate()))

            # In order to start receiving images, enable image transfer. When image transfer is disabled,
            # the vector of images returned by Tracker.get_image() will be empty.
            tracker.enable_image_transfer()

            # The standard PST trackers will run at a reduced frame rate of 30 Hz when image transfer is enabled.
            # However, since this frame rate is temporary for as long as image transfer is enabled, that frame rate
            # will not be reported as the current frame rate.
            print("Enabled image transfer. Current frame rate: " + str(tracker.get_framerate()))
            print("***************************\n")

            # Try to capture 100 images.
            for i in range(100):
                # Try to get the last grabbed image.
                # Note that enabling image transfer takes some time. While image transfer is being enabled,
                # the images list in the Image object will be empty.
                image = tracker.get_image()

                if image is not None:
                    print("Retrieval operation successful!\n")
                    print("Retrieved " + str(len(image.images)) + " image(s) of size: " + str(image.width) +  " X " + str(image.height) + "\n")
                    # Do something with the image
                else:
                    print("Retrieval operation unsuccessful!\n")

                # Don't request images too fast, wait for around 1/60 seconds.
                time.sleep(0.016)

            # Wait for 5 seconds, since this is > 4 seconds, image transfer will be disabled automatically.
            print("Waiting 5 seconds for image transfer to automatically be disabled...\n")
            time.sleep(5)

            # Try to grab one image. Since image retrieval timed out, it should return an empty image vector.
            image = tracker.get_image()
            if image is not None:
                print("Retrieval operation successful!\n")
                print("Retrieved " + str(len(image.images)) + " image(s) of size: " + str(image.width) +  " X " + str(image.height) + "\n")
            else:
                print("Retrieval operation unsuccessful!\n")

            while running:
                time.sleep(0.1)

    except psterrors.TrackerError as err:
        # Catch TrackerError and print error messages.
        print(err.message)

if __name__ == "__main__":
    main()
