/**
 * trackingtarget.cpp
 * Copyright PS-Tech B.V. All Rights Reserved.
 *
 * This example shows how to work with tracking targets using the PST SDK.
 * Note that at this moment tracking targets can not be trained or 
 * imported using the PST SDK. In order to add new tracking targets, please
 * use the PST Client together with the PSTech::pstsdk::Tracker::EnableSharedMemory() 
 * function, or use the stand-alone PST Server to configure the tracking targets.
 *
 */

#ifdef WIN32
#include <windows.h>
#else
#include <csignal>
#endif

/*
 * Define handler functions required to ensure a clean shutdown of the PST Tracker when the
 * application is terminated.
 */

static void Exithandler(int sig);

#ifdef WIN32
BOOL WINAPI ConsoleHandler(DWORD CEvent)
{
    Exithandler(CEvent);
    return TRUE;
}
#endif

/* End of handler functions */

#include <iostream>
#include <thread>
#include <chrono>
#include <atomic>

#include "pstsdk_cpp.h"
#include "TrackerExceptions.h"
#include "PstStringIoStream.h"

/* Control variable for main loop */
static std::atomic<bool> running(true);

/* Number of data points to grab before application termination */
static const uint32_t numberOfSamplesToGrab = 100;

/*
 * Implementation of the PSTech::pstsdk::Listener class to receive tracking data.
 * The OnTrackerData() callback function receives the data as soon as it becomes 
 * available and prints the tracking target name to the command line.
 */
class MyListener : public PSTech::pstsdk::Listener
{    
    virtual void OnTrackerData(const PSTech::pstsdk::TrackerData& td)
    {
        static uint32_t samplesGrabbed = 0;
        if (samplesGrabbed++ >= numberOfSamplesToGrab)
            running = false;

        for (const PSTech::pstsdk::TargetPose& pose : td.targetlist)
            std::cout << "Detected " << pose.name << "\n";
    }
} listener;

/*
 * Implement the exit handler to shut-down the PST Tracker connection on application termination.
 */
static void Exithandler(int sig)
{
    PSTech::pstsdk::Tracker::Shutdown();
    running = false;
}

int main(int argc, char *argv[])
{
    // Register the exit handler with the application
#ifdef WIN32
    SetConsoleCtrlHandler((PHANDLER_ROUTINE)ConsoleHandler, TRUE);
#else
    signal(SIGTERM, Exithandler);
    signal(SIGKILL, Exithandler);
    signal(SIGQUIT, Exithandler);
    signal(SIGINT, Exithandler);
#endif

    // Implement error handling of PSTech::TrackerException exceptions to prevent 
    // improper PST Tracker shutdown on errors.
    try
    {
        // Create an instance of the Tracker object using the default configuration path and file names.
#ifdef WIN32
        PSTech::pstsdk::Tracker pst;
#else
        // On Linux, specify the type of grabber that needs to be used as the last parameter: 
        // "basler_ace" for PST HD or "basler_dart" for PST Pico
        PSTech::pstsdk::Tracker pst("","config.cfg","models.db",argv[1]);
#endif

        // Print version number of the tracker server being used.
        std::cout << "Running PST Server version " << pst.GetVersionInfo() << "\n";

        // Register the listener object to the tracker server.
        pst.AddTrackerListener(&listener);
        
        std::cout << "Put the Reference card in front of the PST in order to see tracking results.\n\n";
        
        // Start the tracker server.
        pst.Start();
        
        // Perform a system check to see if the tracker server is running OK and print the result.
        std::cout << "System check: " << (int)pst.Systemcheck() << "\n";
        
        // Retrieve the list of registered tracking targets and print their names and current status (active or not).
        PSTech::pstsdk::TargetStatuses targets = pst.GetTargetList();
        std::cout << "Found " << targets.size() << " tracking targets:\n";
        for (PSTech::pstsdk::TargetStatus& t : targets)
            std::cout << t.name << "\t" << t.status << "\n";
        std::cout << "\n";

        // Enable the Reference target. Note that this will cause a PSTech::TrackerException in case the Reference
        // target has not been created. The Reference target can be created using the PST Client.
        pst.SetTargetStatus("Reference", true);

        // Get the 3D marker positions making up the Reference device and display them.
        // Note that this will cause a PSTech::TrackerException in case the Reference target has not been created.
        PSTech::pstsdk::MarkerList markers = pst.GetTargetMarkers("Reference");
        std::cout << "3D marker positions making up the Reference target:\n";
        for (auto& marker : markers)
            std::cout << "x: " << marker[0] << "\t" << "y: " << marker[1] << "\t" << "z: " << marker[2] << "\n";
        std::cout << "\n";
        
        // Main loop, wait for auto-termination.
        while (running)
        {
            std::this_thread::sleep_for(std::chrono::milliseconds(100));
        }
    }
    catch (PSTech::TrackerException &e)
    {
        // Catch PSTech::TrackerException exceptions and print error messages.
        std::cout << e.full_description() << "\n";        
    }
    
    // Make sure that the connection to the PST Tracker is shut down properly.
    PSTech::pstsdk::Tracker::Shutdown();

    // Pause command line to see results.
    std::cout << "Press enter to continue...\n";
    std::cin.get();

    return 0;
}