/**
 * sharedmemory.cpp
 * Copyright PS-Tech B.V. All Rights Reserved.
 *
 * This example shows how to activate the shared memory communication pipeline
 * that enables communication of the PST Client application through the PST SDK.
 * Note that for the shared memory pipeline to function, the application has to
 * run with elevated access (administrator rights). After enabling shared memory,
 * the PST Client application can be used to download calibration information and
 * manage tracking targets.
 * 
 */

#ifdef WIN32
#include <windows.h>
#else
#include <csignal>
#endif

/*
 * Define handler functions required to ensure a clean shutdown of the PST Tracker
 * when the application is terminated.
 */

static void Exithandler(int sig);

#ifdef WIN32
BOOL WINAPI ConsoleHandler(DWORD CEvent)
{
    Exithandler(CEvent);
    return TRUE;
}
#endif

/* End of handler functions */

#include <iostream>
#include <thread>
#include <chrono>

#include "pstsdk_cpp.h"
#include "TrackerExceptions.h"

/*
 * Implement the exit handler to shut-down the PST Tracker connection on application termination.
 */
static void Exithandler(int sig)
{
    PSTech::pstsdk::Tracker::DisableSharedMemory();
    PSTech::pstsdk::Tracker::Shutdown();
    exit(0);
}

int main(int argc, char *argv[])
{
    // Register the exit handler with the application
#ifdef WIN32
    SetConsoleCtrlHandler((PHANDLER_ROUTINE)ConsoleHandler, TRUE);
#else
    signal(SIGTERM, Exithandler);
    signal(SIGKILL, Exithandler);
    signal(SIGQUIT, Exithandler);
    signal(SIGINT, Exithandler);
#endif

    // Implement error handling of PSTech::TrackerException exceptions to prevent 
    // improper PST Tracker shutdown on errors.
    try
    {
        // Create an instance of the Tracker object using the default configuration path and file names.
#ifdef WIN32
        PSTech::pstsdk::Tracker pst;
#else
        // On Linux, specify the type of grabber that needs to be used as the last parameter: 
        // "basler_ace" for PST HD or "basler_dart" for PST Pico
        PSTech::pstsdk::Tracker pst("","config.cfg","models.db",argv[1]);
#endif
        // Start the tracker server.
        pst.Start();

        // Enable the shared memory pipeline to enable client connections.
        pst.EnableSharedMemory();

        std::cout << "Shared memory server initialized. Start the PST Client application to create a connection.\n";
        
        // Wait for one minute before terminating this application.
        std::this_thread::sleep_for(std::chrono::milliseconds(60000));

        // Disable the shared memory pipeline. If the PST Client is still connected at this point,
        // it will try to reconnect and otherwise exit.
        pst.DisableSharedMemory();
    }
    catch (PSTech::TrackerException &e)
    {
        std::cout << e.full_description() << "\n";
        // Handle the error.
    }

    // Make sure that the connection to the PST Tracker is shut down properly.
    PSTech::pstsdk::Tracker::Shutdown();

    return 0;
}