/**
 * restserver.c
 * Copyright PS-Tech B.V. All Rights Reserved.
 *
 * This example shows how to enable the REST server using the PST SDK.
 * The REST server enables network-based access to the PST Tracker using
 * the HTTP protocol. GET and POST requests can be made to the server to
 * send and receive data and change parameters. The REST interface offers
 * a programming language independent interface to the PST Tracker.
 * Besides accessing the REST server directly using a browser, a networking
 * library like Curl can be used to interface with the server programatically.
 *
 */

#ifdef WIN32
#include <windows.h>
#else
#include <unistd.h>
#endif

/*
 * Define handler functions required to ensure a clean shutdown of the PST Tracker
 * when the application is terminated.
 */

static void Exithandler(int sig);

#ifdef WIN32
BOOL WINAPI ConsoleHandler(DWORD CEvent)
{
    Exithandler(CEvent);
    return TRUE;
}
#endif

/* End of handler functions */

#include "pstsdk_c.h"
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

/*
* Implement the exit handler to shut-down the PST Tracker connection on application termination.
*/
static void Exithandler(int sig)
{
    pst_sdk_disable_rest_server();
    pst_sdk_shutdown();
    exit(0);
}

// Print the last error message.
void PrintLastErrorMessage()
{
    char* last_error_message = NULL;
    EPstErrorStatus error_status = pst_alloc_and_get_last_error_message(&last_error_message);
    if (error_status != PST_ERROR_STATUS_OK)
    {
        last_error_message = "Failed to allocate memory error.";
    }
    printf("last error message: %s \n", last_error_message);
    pst_free(last_error_message);
}

// Check error status and shutdown tracker upon error.
void CheckErrorCode(EPstErrorStatus status)
{
    if (status != PST_ERROR_STATUS_OK)
    {
        PrintLastErrorMessage();
        pst_sdk_shutdown();
        exit(status);
    }
}

int main(int argc, char *argv[])
{
    // Register the exit handler with the application
#ifdef WIN32
    SetConsoleCtrlHandler((PHANDLER_ROUTINE)ConsoleHandler, TRUE);
#else
    signal(SIGTERM, Exithandler);
    signal(SIGKILL, Exithandler);
    signal(SIGQUIT, Exithandler);
    signal(SIGINT, Exithandler);
#endif

    PstTracker ctracker;

#ifdef WIN32
    // Create an instance of the Tracker object using the default configuration path and file names.
    CheckErrorCode(pst_tracker_init(&ctracker));
#else
    // On Linux, specify the type of grabber that needs to be used as the last parameter:
    // "basler_ace" for PST HD or "basler_dart" for PST Pico
    CheckErrorCode(pst_tracker_init4(&ctracker, "", "config.cfg", "models.db", argv[1]));
#endif

    // Start the tracker server.
    CheckErrorCode(pst_tracker_start(&ctracker));

    // Enable the REST server. To check if the REST server is started correctly, browse to
    // http://localhost:7278/PSTapi/SystemCheck
    // In order to use 127.0.0.1 an an address on Windows 7, execute the following command
    // on an elevated command prompt to allow communication to this address:
    // "netsh http add urlacl url=http://127.0.0.1:7278/ user=EVERYONE listen=yes delegate=no"
    CheckErrorCode(pst_sdk_enable_rest_server("localhost", "7278", 3000));

    printf("PST REST server enabled. See the PST SDK Manual for example commands.\n");

    // Wait for one minute before terminating this application.
#ifdef WIN32
    Sleep(60000);
#else
    sleep(60);
#endif

    // Disable the REST server
    pst_sdk_disable_rest_server();

    // Make sure that the connection to the PST Tracker is shut down properly.
    pst_tracker_destroy(&ctracker);

    return 0;
}
